package cn.com.duibaboot.ext.autoconfigure.cat;

import java.io.IOException;
import java.util.concurrent.Future;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.internal.DefaultTransaction;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.nio.protocol.HttpAsyncResponseConsumer;
import org.apache.http.protocol.HttpContext;

import org.springframework.beans.BeansException;

/**
 * 让CloseableHttpAsyncClient支持Cat监控
 */
public class CatHttpAsyncClientPostProcessor implements SpecifiedBeanPostProcessor<CloseableHttpAsyncClient> {
    private static final String CAT_HTTP_ASYNC_CLIENT_NAME = "HttpAsyncClient";

    @Override
    public Class<CloseableHttpAsyncClient> getBeanType() {
        return CloseableHttpAsyncClient.class;
    }

    @Override
    public Object postProcessBeforeInitialization(CloseableHttpAsyncClient bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(CloseableHttpAsyncClient bean, String beanName) throws BeansException {
        return new CatHttpAsyncClientWrapper(bean);
    }

    @Override
    public int getOrder() {
        return 0;
    }

    /**
     * CloseableHttpAsyncClient的包装类，让CloseableHttpAsyncClient支持Cat监控。
     */
    public static class CatHttpAsyncClientWrapper extends CloseableHttpAsyncClient {

        private CloseableHttpAsyncClient original;

        public CatHttpAsyncClientWrapper(CloseableHttpAsyncClient original){
            this.original = original;
        }

        private <T> FutureCallback<T> getCatWrappedCallback(FutureCallback<T> callback, String hostname){
            if(!CatUtils.isCatEnabled()){
                return callback;
            }

            long start = System.nanoTime();
            FutureCallback<T> wrappedCallback = new FutureCallback<T>() {
                @Override
                public void completed(T result) {
                    DefaultTransaction transaction = (DefaultTransaction) Cat.newTransaction(CAT_HTTP_ASYNC_CLIENT_NAME, hostname);
                    transaction.setDurationStart(start);
                    transaction.setStatus(Message.SUCCESS);
                    transaction.complete();

                    callback.completed(result);
                }

                @Override
                public void failed(Exception ex) {
                    //httpclient异常不记录到cat，不触发告警，以防对外部接口的调用有问题而触发大量告警。
                    //失败单独记录, 但不幸的是，cat上不能直观地看到HttpClient的失败数
                    DefaultTransaction transaction = (DefaultTransaction) Cat.newTransaction(CAT_HTTP_ASYNC_CLIENT_NAME, hostname + "(failed)");
                    transaction.setDurationStart(start);
                    transaction.setStatus(Message.SUCCESS);
                    transaction.complete();

                    callback.failed(ex);
                }

                @Override
                public void cancelled() {
                    //httpclient异常不记录到cat，不触发告警，以防对外部接口的调用有问题而触发大量告警。
                    //失败单独记录, 但不幸的是，cat上不能直观地看到HttpClient的失败数
                    DefaultTransaction transaction = (DefaultTransaction) Cat.newTransaction(CAT_HTTP_ASYNC_CLIENT_NAME, hostname + "(cancelled)");
                    transaction.setDurationStart(start);
                    transaction.setStatus(Message.SUCCESS);
                    transaction.complete();

                    callback.cancelled();
                }
            };
            return wrappedCallback;
        }

        @Override
        public <T> Future<T> execute(HttpAsyncRequestProducer requestProducer, HttpAsyncResponseConsumer<T> responseConsumer, HttpContext context, FutureCallback<T> callback) {
            return original.execute(requestProducer, responseConsumer, context, getCatWrappedCallback(callback, getHostName(requestProducer)));
        }

        private String getHostName(HttpAsyncRequestProducer requestProducer) {
            String hostname = "unknown";
            HttpHost host = requestProducer.getTarget();
            if (host != null) {
                hostname = host.getHostName();
            }
            return hostname;
        }

        private String getHostName(HttpHost target, HttpRequest request) {
            String hostname = "unknown";
            if (target != null) {
                hostname = target.getHostName();
            } else if(request != null && request instanceof HttpUriRequest) {
                hostname = ((HttpUriRequest)request).getURI().getHost();
            }

            return hostname;
        }

        @Override
        public boolean isRunning() {
            return original.isRunning();
        }

        @Override
        public void start() {
            original.start();
        }

        @Override
        public <T> Future<T> execute(HttpAsyncRequestProducer requestProducer, HttpAsyncResponseConsumer<T> responseConsumer, FutureCallback<T> callback) {
            return original.execute(requestProducer, responseConsumer, getCatWrappedCallback(callback, getHostName(requestProducer)));
        }

        @Override
        public Future<HttpResponse> execute(HttpHost target, HttpRequest request, HttpContext context, FutureCallback<HttpResponse> callback) {
            return original.execute(target, request, context, getCatWrappedCallback(callback, getHostName(target, request)));
        }

        @Override
        public Future<HttpResponse> execute(HttpHost target, HttpRequest request, FutureCallback<HttpResponse> callback) {
            return original.execute(target, request, getCatWrappedCallback(callback, getHostName(target, request)));
        }

        @Override
        public Future<HttpResponse> execute(HttpUriRequest request, HttpContext context, FutureCallback<HttpResponse> callback) {
            return original.execute(request, context, getCatWrappedCallback(callback, getHostName(null, request)));
        }

        @Override
        public Future<HttpResponse> execute(HttpUriRequest request, FutureCallback<HttpResponse> callback) {
            return original.execute(request, getCatWrappedCallback(callback, getHostName(null, request)));
        }

        @Override
        public void close() throws IOException {
            original.close();
        }
    }
}
