package cn.com.duibaboot.ext.autoconfigure.datasource.encrypt;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.datasource.encrypt.config.DBEncryptPolicy;
import cn.com.duibaboot.ext.autoconfigure.datasource.encrypt.config.DuibaDBEncryptProperties;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.shardingsphere.encrypt.api.EncryptRuleConfiguration;
import org.apache.shardingsphere.encrypt.api.EncryptTableRuleConfiguration;
import org.apache.shardingsphere.encrypt.api.EncryptorRuleConfiguration;
import org.apache.shardingsphere.encrypt.rule.EncryptRule;
import org.apache.shardingsphere.shardingjdbc.jdbc.core.datasource.EncryptDataSource;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanInitializationException;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.sql.DataSource;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * 数据源的初始化后置操作类
 * 在 DataSource 初始化结束的时候，判断如果该数据源有相关的加密配置，则通过包装的方式，把这个数据源包装到 EncryptDataSource 中
 * Created by gyf .
 * 2020/7/24 .
 */
@Slf4j
public class EncryptDataSourceSpecifiedBeanPostProcessor implements SpecifiedBeanPostProcessor {

    private final Map<String, DBEncryptConfig>        dbEncryptConfigMap                  = new HashMap<>();
    private final EncryptRuleConfigurationTransformer encryptRuleConfigurationTransformer = new EncryptRuleConfigurationTransformer();

    @Resource
    private DuibaDBEncryptProperties duibaDBEncryptProperties;

    @PostConstruct
    public void init() {
        Map<String, EncryptorRuleConfiguration> encryptors = encryptRuleConfigurationTransformer.transformEncryptors(duibaDBEncryptProperties.getEncryptors());
        Map<String, DBEncryptPolicy> policies = duibaDBEncryptProperties.getPolicy();
        if (encryptors.isEmpty() || policies.isEmpty()) {
            return;
        }

        Properties defaultProps = duibaDBEncryptProperties.getProps();
        for (Map.Entry<String, DBEncryptPolicy> policyEntry : policies.entrySet()) {
            DBEncryptPolicy policy = policyEntry.getValue();
            List<String> datasources = policy.getDatasources();
            if (CollectionUtils.isEmpty(datasources)) {
                throw new IllegalStateException("数据库加密配置错误，policy: " + policyEntry.getKey() + " 未配置数据源");
            }
            Map<String, EncryptTableRuleConfiguration> tables = encryptRuleConfigurationTransformer.transformTables(policy.getTables());
            if (tables.isEmpty()) {
                throw new IllegalStateException("数据库加密配置错误，policy: " + policyEntry.getKey() + " 未配置数据库表加密关系");
            }
            for (String ds : datasources) {
                EncryptRuleConfiguration encryptRuleConfiguration = encryptRuleConfigurationTransformer.transformEncryptRuleConfiguration(encryptors, tables);
                if (dbEncryptConfigMap.get(ds) != null) {
                    throw new IllegalStateException("数据库加密配置错误，同一个数据源不能配置在多个加密规则中");
                }
                dbEncryptConfigMap.put(ds, new DBEncryptConfig(ds, encryptRuleConfiguration, this.mergeProps(policy.getProps(), defaultProps)));
            }
        }
    }

    @Override
    public Class<DataSource> getBeanType() {
        return DataSource.class;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (dbEncryptConfigMap.isEmpty()) {
            return bean;
        }

        DBEncryptConfig dbEncryptConfig = dbEncryptConfigMap.get(beanName);
        if (dbEncryptConfig == null) {
            return bean;
        }
        try {
            return new EncryptDataSource((DataSource) bean, new EncryptRule(dbEncryptConfig.getEncryptRuleConfiguration()), dbEncryptConfig.getProps());
        } catch (SQLException e) {
            throw new BeanInitializationException("包装EncryptDataSource异常", e);
        }
    }

    @Override
    public int getOrder() {
        return 100;
    }

    /**
     * 合并policy中的【自定义配置】 以及【全局配置】
     * 如果没有【自定义配置】，那么 返回【全局配置】
     * 如果有【自定义配置】，那么把【自定义配置】覆盖【全局配置】后再返回
     * @param customizeProps
     * @param defaultProps
     * @return
     */
    private Properties mergeProps(Properties customizeProps, Properties defaultProps) {
        if (customizeProps.isEmpty()) {
            return defaultProps;
        }
        Properties newProp = new Properties();
        newProp.putAll(defaultProps);
        for (Map.Entry<Object, Object> entry : customizeProps.entrySet()) {
            newProp.put(entry.getKey(), entry.getValue());
        }
        return newProp;
    }

}
