package cn.com.duibaboot.ext.autoconfigure.logger;

import ch.qos.logback.classic.AsyncAppender;
import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.joran.JoranConfigurator;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.ConsoleAppender;
import ch.qos.logback.core.FileAppender;
import ch.qos.logback.core.encoder.Encoder;
import ch.qos.logback.core.pattern.PatternLayoutEncoderBase;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DuibaEurekaAutoServiceRegistration;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.RibbonCustomAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.duibaboot.ext.autoconfigure.core.utils.SpringBootUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.hazelcast.EurekaOneDiscoveryStrategy;
import cn.com.duibaboot.ext.autoconfigure.logger.filter.CatLoggerCountFilter;
import cn.com.duibaboot.ext.autoconfigure.logger.filter.DenyFilter;
import cn.com.duibaboot.ext.autoconfigure.logger.filter.LoggerSizeFilter;
import cn.com.duibaboot.ext.autoconfigure.logger.filter.ThreadNameFilter;
import cn.com.duibaboot.ext.autoconfigure.logger.logback.appender.KafkaAppender;
import cn.com.duibaboot.ext.autoconfigure.logger.logback.appender.KafkaAppenderProperties;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.actuate.endpoint.EndpointId;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnResource;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.diagnostics.LoggingFailureAnalysisReporter;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ApplicationContextEvent;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.io.ClassPathResource;
import org.springframework.kafka.core.KafkaTemplate;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;

/**
 * Created by xugf on 2016/11/25
 */
@Configuration
@ConditionalOnClass({LoggerContext.class})
public class LoggerAutoConfiguration {

	private static final Logger logger = LoggerFactory.getLogger(LoggerAutoConfiguration.class);

	private static final String APPLICATION_LOG = "application.log";

//	/**
//	 * 附加logback配置文件(for loginsight日志监控)
//	 * @return
//	 */
//	@Bean
//	@ConditionalOnResource(resources="classpath:logback/loginsight-logback.xml")
//	public ApplicationListener loginsightLoggerConfiguar() {
//		return new ApplicationListener<ContextRefreshedEvent>(){
//
//			private boolean flag = true;
//
//			@Override
//			public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
//				if(flag){
//					try {
//						LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
//						JoranConfigurator configurator = new JoranConfigurator();
//						configurator.setContext(loggerContext);
//						//logback 添加新的配置文件
//						configurator.doConfigure(new ClassPathResource("/logback/loginsight-logback.xml").getInputStream());
//
//						Appender loginsightAppender = loggerContext.getLogger("loginsight_log").getAppender("ASYNC_LOGINSIGHT_APPENDER");
//						loggerContext.getLogger(ch.qos.logback.classic.Logger.ROOT_LOGGER_NAME).addAppender(loginsightAppender);
//					} catch (Exception e) {
//						logger.error(e.getMessage(), e);
//					}
//					flag = false;
//				}
//			}
//		};
//	}

	/**
	 * 附加logback配置文件(for 引流回归)
	 * @return
	 */
	@Bean
	@ConditionalOnResource(resources="/logback/flowreplay-logback.xml")
	public ApplicationListener FlowReplayAppenderConfiguration() {	// NOSONAR
		return new ApplicationListener<ContextRefreshedEvent>(){

			private boolean flag = true;

			@Override
			public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
				if (flag) {
					try {
						LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
						JoranConfigurator configurator = new JoranConfigurator();
						configurator.setContext(loggerContext);
						//logback 添加新的配置文件
						configurator.doConfigure(new ClassPathResource("/logback/flowreplay-logback.xml").getInputStream());

						ch.qos.logback.classic.Logger logger = loggerContext.getLogger(ch.qos.logback.classic.Logger.ROOT_LOGGER_NAME);

						Appender recordAppender = loggerContext.getLogger("record_log").getAppender("ASYNC_RECORD_APPENDER");
						logger.addAppender(recordAppender);

						if (FlowReplayUtils.isReplayEnv()) {
							Appender replayAppender = loggerContext.getLogger("replay_log").getAppender("ASYNC_REPLAY_APPENDER");
							logger.addAppender(replayAppender);
							this.removeApplicationLogAppender(loggerContext.getLoggerList());
						}
					} catch (Exception e) {
						logger.error(e.getMessage(), e);
					}
					flag = false;
				}
			}

			private void removeApplicationLogAppender(List<ch.qos.logback.classic.Logger> loggers) {
				for (ch.qos.logback.classic.Logger logger : loggers) {

					Iterator<Appender<ILoggingEvent>> iter = logger.iteratorForAppenders();

					List<Appender<ILoggingEvent>> appenders2Remove = new ArrayList<>();

					while (iter.hasNext()) {
						Appender<ILoggingEvent> appender = iter.next();
						if (appender instanceof FileAppender) {//如果是同步的FileAppender，则转换为异步输出Appender
							FileAppender fileAppender = (FileAppender) appender;
							if (fileAppender.getFile().contains(APPLICATION_LOG)) {
								appenders2Remove.add(fileAppender);
							}
						}
						if (appender instanceof AsyncAppender) {
							Iterator<Appender<ILoggingEvent>> iterator = ((AsyncAppender) appender).iteratorForAppenders();
							while (iterator.hasNext()) {
								Appender<ILoggingEvent> childAsyncAppender = iterator.next();
								if (childAsyncAppender instanceof FileAppender) {//如果是同步的FileAppender，则转换为异步输出Appender
									FileAppender childAsyncFileAppender = (FileAppender) childAsyncAppender;
									if (childAsyncFileAppender.getFile().contains(APPLICATION_LOG)) {
										appenders2Remove.add(appender);
									}
								}
							}
						}
					}

					for (Appender<ILoggingEvent> syncAppender : appenders2Remove) {
						logger.detachAppender(syncAppender);
					}
				}

			}

		};
	}

	/**
	 * 每隔5秒把所有日志中的outputstream进行flush。使用户尽快看到输出的日志
	 * @return
	 */
	@Bean
	public ApplicationListener duibaLogbackFlusher() {
		return new DuibaLogbackFlusher();
	}

	private static class DuibaLogbackFlusher implements ApplicationListener<ContextRefreshedEvent>{
		private Timer timer = new Timer("logback-flusher", true);
		private Set<FileAppender> fileAppendersNeedFlush = Collections.synchronizedSet(new HashSet<>());
		private static final int intevalPeriod = 5 * 1000;//second
		private static final int timeDelay = 5 * 1000;
		private boolean flag = true;
		TimerTask timerTask = new TimerTask() {
			@Override
			public void run(){
				fileAppendersNeedFlush.stream().map(fileAppender -> fileAppender.getOutputStream())
						.filter(o -> o!= null)
						.forEach(outputStream -> {
					try {
						outputStream.flush();
					} catch (IOException e) {
						logger.error(e.getMessage(),e);
					}
				});
			}
		};
		@Override
		public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {//NOSONAR
			if(!flag) {
				return;
			}
			LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
			List<ch.qos.logback.classic.Logger> loggers = loggerContext.getLoggerList();

			for(ch.qos.logback.classic.Logger logger : loggers) {
				Iterator<Appender<ILoggingEvent>> iter = logger.iteratorForAppenders();

				while(iter.hasNext()){
					Appender<ILoggingEvent> appender = iter.next();
					if(appender instanceof AsyncAppender){
						AsyncAppender asyncAppender = (AsyncAppender)appender;
						Iterator<Appender<ILoggingEvent>> iteratorForAppenders = asyncAppender.iteratorForAppenders();
						if(iteratorForAppenders.hasNext()){
							appender = iteratorForAppenders.next();
						}
					}
					if(appender instanceof FileAppender){
						final FileAppender fileAppender = (FileAppender)appender;

						fileAppendersNeedFlush.add(fileAppender);
					}
				}
			}
			timer.schedule(timerTask,timeDelay,intevalPeriod);
			flag = false;
		}
	};

	/**
	 * 强制改变部分日志的日志级别。
	 *
	 * @return
	 */
	@Bean
	public ApplicationListener loggerLevelChangerApplicationListener() {

		return new ApplicationListener<ContextRefreshedEvent>(){
			private boolean flag = true;
			private Map<String, Level> loggerNames = Maps.newHashMap();

			{
			    // ShardingSphere-SQL showsql的开关打开后，需要打印日志
				loggerNames.put("ShardingSphere-SQL", Level.INFO);
				//HealthIndicator相关类，当/health检查失败会打印warn日志
				loggerNames.put("org.springframework.boot.actuate.health", Level.WARN);
				loggerNames.put("org.springframework.cloud", Level.WARN);
				loggerNames.put("com.hazelcast.internal.cluster.ClusterService", Level.ERROR);
				//强制输错启动失败的错误
				loggerNames.put(LoggingFailureAnalysisReporter.class.getName(), Level.DEBUG);
				loggerNames.put(DuibaEurekaAutoServiceRegistration.class.getName(), Level.INFO);
				loggerNames.put(RibbonCustomAutoConfiguration.class.getName(), Level.INFO);
				//强制lettuce打印断网重连日志
				loggerNames.put("io.lettuce", Level.INFO);
				loggerNames.put(EurekaOneDiscoveryStrategy.class.getName(), Level.INFO);
				loggerNames.put(DuibaEurekaAutoServiceRegistration.class.getName(), Level.INFO);
				loggerNames.put(EndpointId.class.getName(), Level.ERROR);
			}

			@Override
			public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
				if(flag){
					for(Map.Entry<String, Level> entry : loggerNames.entrySet()) {
						ch.qos.logback.classic.Logger logger = ((ch.qos.logback.classic.Logger) LoggerFactory.getLogger(entry.getKey()));
						logger.setLevel(entry.getValue());
					}

					flag = false;
				}

			}
		};
	}

	/**
	 * 自动把logback日志输出转化为异步输出(如果已经配置异步输出则忽略)
	 * @return
	 */
	@Bean
	public ApplicationListener asyncLoggerConfiguar() {
		return new AsyncLoggerConfiguarListener();
	}

	@Bean
    public LoggerSizeFilter loggerSizeFilter(){
	    return new LoggerSizeFilter();
    }

	private static class AsyncLoggerConfiguarListener implements ApplicationListener<ContextRefreshedEvent>{

		private boolean flag = true;
		//这个map用于缓存日志同步appender到日志异步appender
		private Map<FileAppender, AsyncAppender> syncAppender2asyncAppenderCachedMap = new HashMap<>();
		//用Set保证多个Logger注入同一个Appender时，FileAppender不会重复注入计数器
		private Set<FileAppender> fileAppenders = Sets.newHashSet();
		private Set<ConsoleAppender> consoleAppenders = Sets.newHashSet();
		private Set<AsyncAppender> asyncAppenders = Sets.newHashSet();

		@Resource
        LoggerSizeFilter loggerSizeFilter;

		@Override
		public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
			if(!flag) {
				return;
			}
			try {
				LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();

				//ch.qos.logback.classic.Logger rootLog = loggerContext.getLogger(ch.qos.logback.classic.Logger.ROOT_LOGGER_NAME);
				List<ch.qos.logback.classic.Logger> loggers = loggerContext.getLoggerList();
				for(ch.qos.logback.classic.Logger logger : loggers) {
					switchLogAppend2Async(logger);
				}
				//只有设置了ImmediateFlush为false，才不会每条日志都刷盘，要不要设置？
//				for (FileAppender fileAppender : fileAppenders) {
//					fileAppender.setImmediateFlush(false);
//				}

				for (FileAppender fileAppender : fileAppenders) {
					fileAppender.addFilter(loggerSizeFilter);//日志打印单条大小限制,或者每秒打印条数过多告警
				}

				for (ConsoleAppender consoleAppender : consoleAppenders) {
					consoleAppender.addFilter(new ThreadNameFilter());//设置线程名字
					if (SpringBootUtils.isJarInJarMode()) {
						consoleAppender.addFilter(new DenyFilter());//jar模式运行时，日志不打印到标准输出，节省一些性能
					}
				}

				//给file.log文件增加计数
				if(CatUtils.isCatClassExists()) {
					for (FileAppender fileAppender : fileAppenders) {
						fileAppender.addFilter(new CatLoggerCountFilter(fileAppender.getFile()));//增加文件日志计数
					}
				}

				//这个map已经没用了，赋值null以便jvm回收
				syncAppender2asyncAppenderCachedMap = null;
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
			}
			flag = false;
		}

		/**
		 * 把当前log的同步输出appender转化为异步日志appender
		 * @param log
		 */
		private void switchLogAppend2Async(ch.qos.logback.classic.Logger log){ //NOSONAR
			Iterator<Appender<ILoggingEvent>> iter = log.iteratorForAppenders();
			List<Appender<ILoggingEvent>> asyncAppenders2Add = new ArrayList<>();
			List<Appender<ILoggingEvent>> appenders2Remove = new ArrayList<>();

			while (iter.hasNext()) {
				Appender<ILoggingEvent> appender = iter.next();
				if (appender instanceof FileAppender){//如果是同步的FileAppender，则转换为异步输出Appender
					FileAppender fileAppender = (FileAppender)appender;
					AsyncAppender asyncAppender = getCachedAsyncAppender(fileAppender);
					asyncAppenders2Add.add(asyncAppender);
					appenders2Remove.add(appender);
					fileAppenders.add(fileAppender);
				}
				if (appender instanceof ConsoleAppender){
					ConsoleAppender consoleAppender = (ConsoleAppender)appender;
					consoleAppenders.add(consoleAppender);
				}
				if (appender instanceof AsyncAppender){
					Iterator<Appender<ILoggingEvent>> iterator = ((AsyncAppender) appender).iteratorForAppenders();
					FileAppender fileAppender = null;
					while (iterator.hasNext()){
						Appender<ILoggingEvent> childAsyncAppender = iterator.next();
						if(childAsyncAppender instanceof FileAppender){
							fileAppenders.add((FileAppender)childAsyncAppender);
							fileAppender = (FileAppender)childAsyncAppender;
						}
					}

					if (!asyncAppenders.contains(appender)){
						if (fileAppender != null) {
							setIncludeCallerData(fileAppender, (AsyncAppender) appender);
						}
						appender.addFilter(new ThreadNameFilter());//设置线程名字
					}

					asyncAppenders.add((AsyncAppender)appender);
				}
			}

			for (Appender<ILoggingEvent> syncAppender : appenders2Remove) {
				log.detachAppender(syncAppender);
			}
			for (Appender<ILoggingEvent> asyncAppender : asyncAppenders2Add) {
				log.addAppender(asyncAppender);
			}
		}

		/**
		 * 获得缓存的异步输出appender，如果未缓存，则构造之
		 * @return
		 */
		private AsyncAppender getCachedAsyncAppender(FileAppender fileAppender){
			AsyncAppender asyncAppender = syncAppender2asyncAppenderCachedMap.get(fileAppender);
			if(asyncAppender == null){
				asyncAppender = new AsyncAppender();
				asyncAppender.setContext((LoggerContext)LoggerFactory.getILoggerFactory());
				//不丢失日志. 默认地,如果队列的80%已满,则会丢弃TRACT、DEBUG、INFO级别的日志
				asyncAppender.setDiscardingThreshold(0);
				//更改默认的队列的深度,该值会影响性能.默认值为256
				asyncAppender.setQueueSize(512);
				//名字加入前缀

				setIncludeCallerData(fileAppender, asyncAppender);
				asyncAppender.addFilter(new ThreadNameFilter());//设置线程名字

				asyncAppender.setName("DuibaBootAutoAsync_"+fileAppender.getName());
				asyncAppender.addAppender(fileAppender);
				//这里只管start，LoggerContext.stop()被调用时会自动调用这个asyncAppender的stop方法。
				asyncAppender.start();

				syncAppender2asyncAppenderCachedMap.put(fileAppender, asyncAppender);
			}

			return asyncAppender;
		}

		private void setIncludeCallerData(FileAppender fileAppender, AsyncAppender asyncAppender) {//性能优化，只在必要的时候设置includeCallerData（经测试，输出行数等信息会增加10倍的rt）
				boolean includeCallerData = false;
				Encoder encoder = fileAppender.getEncoder();
				if(encoder != null && encoder instanceof PatternLayoutEncoderBase){
					PatternLayoutEncoderBase pencoder = (PatternLayoutEncoderBase)encoder;
					String pattern = pencoder.getPattern();
					if(pattern.contains("%L") || pattern.contains("%line")
							|| pattern.contains("%C") || pattern.contains("%class")
							|| pattern.contains("%M") || pattern.contains("%method")
							|| pattern.contains("%F") || pattern.contains("%file")
							|| pattern.contains("%caller")){
						includeCallerData = true;
					}
				}
				fileAppender.setImmediateFlush(false);//这样设置才不会每条日志都刷盘

			//这里设为true才能正确打印出%line等信息,如果不需要打印%line，则这里不需要配为true
			asyncAppender.setIncludeCallerData(includeCallerData);
		}

	}

	/**
	 * 附加KafkaAppender
	 */
	@Configuration
	@ConditionalOnClass({KafkaTemplate.class})
	@EnableConfigurationProperties(KafkaAppenderProperties.class)
	public static class KafkaAppenderConfiguration{

		@Autowired
		private KafkaAppenderProperties kafkaAppenderProperties;
		@Autowired
		private KafkaTemplate kafkaTemplate;

		private boolean flag = true;

		@EventListener(ContextRefreshedEvent.class)
		public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent){
			if(!flag) {
				return;
			}
			try {
				if(kafkaAppenderProperties.getPatterns() == null || kafkaAppenderProperties.getPatterns().length == 0){
					return;
				}

				LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
//					Map<String, List<KafkaAppenderProperties.Pattern>> logger2PatternsMap = new HashMap<>();
				Multimap<String, KafkaAppenderProperties.Pattern> logger2PatternsMap = ArrayListMultimap.create();
				for(KafkaAppenderProperties.Pattern pattern : kafkaAppenderProperties.getPatterns()){
					if(StringUtils.isBlank(pattern.getLoggerName())){
						throw new IllegalStateException("config `duiba.logback.appender.kafka[*].logger-name` must not be blank");
					}
					if(StringUtils.isBlank(pattern.getTopic())){
						throw new IllegalStateException("config `duiba.logback.appender.kafka[*].topic` must not be blank");
					}
					logger2PatternsMap.put(pattern.getLoggerName(), pattern);
				}

				for(String loggerName : logger2PatternsMap.keySet()){
					Collection<KafkaAppenderProperties.Pattern> patterns = logger2PatternsMap.get(loggerName);
					if(CollectionUtils.isEmpty(patterns)){
						continue;
					}
					ch.qos.logback.classic.Logger logger = loggerContext.getLogger(loggerName);
					KafkaAppender appender = new KafkaAppender<>(kafkaTemplate, patterns);
					appender.setContext(loggerContext);
					appender.start();
					logger.addAppender(appender);
				}

			} catch (Exception e) {
				logger.error(e.getMessage(), e);
			}finally {
				flag = false;
			}
		}

	}

	@Resource
	private ApplicationContext applicationContext;

	@EventListener({ContextRefreshedEvent.class, ContextClosedEvent.class})
	public void onEvent(ApplicationContextEvent e){
		if(e instanceof ContextRefreshedEvent) {
			NonRefreshLogbackLoggingSystem.setCanCleanUp(false);
		}else if(e instanceof ContextClosedEvent && e.getApplicationContext() == applicationContext){
			NonRefreshLogbackLoggingSystem.setCanCleanUp(true);
		}
	}

}
