package cn.com.duibaboot.ext.autoconfigure.grouping.dubbo;

import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupUtils;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.rpc.Invocation;
import org.apache.dubbo.rpc.Invoker;
import org.apache.dubbo.rpc.RpcException;
import org.apache.dubbo.rpc.cluster.router.AbstractRouter;

/**
 * 用于服务分组调用的 dubbo路由器
 * 优先调用有相同【服务分组key（多场景）】的服务，如果找不到，那么调用【没有分组key（主场景）】的服务，如果还是没有，那么随便找一台调用
 * Created by gyf .
 * 2020/11/12 .
 */
public class ServiceGroupDubboRouter extends AbstractRouter {

    public ServiceGroupDubboRouter(URL url) {
        this.url = url;
    }

    @Override
    public <T> List<Invoker<T>> route(List<Invoker<T>> invokers, URL url, Invocation invocation) throws RpcException {
        if (CollectionUtils.isEmpty(invokers)) {
            return invokers;
        }

        String currentServiceGroupKey = ServiceGroupContext.getCurrentGroupKey();

        List<Invoker<T>> havePriorityList = new ArrayList<>(invokers.size());  // 优先转发目标服务列表
        List<Invoker<T>> noGroupKeyInvokerList = new ArrayList<>(invokers.size());  // 没有分组标识的服务
        for (Invoker<T> invoker : invokers) {
            String invokerGroupKey = invoker.getUrl().getParameter(ServiceGroupDubboRegistryFactoryWrapper.SERVICE_GROUP_KEY);
            String targetNodeIp = invoker.getUrl().getIp();

            if (this.groupKeyEquals(currentServiceGroupKey, invokerGroupKey, targetNodeIp)) {
                havePriorityList.add(invoker);
            }
            if (StringUtils.isBlank(invokerGroupKey)) {
                noGroupKeyInvokerList.add(invoker);
            }
        }

        // 流量优先转发给具有相同分组标识的
        if (!havePriorityList.isEmpty()) {
            return havePriorityList;
        }
        // 没有找到优先转发的目标服务列表。降级：请求带上了分组标识，并且存在没有分组标识的服务列表，则转发给没有分组标识的服务列表
        if (StringUtils.isNotBlank(currentServiceGroupKey) && !noGroupKeyInvokerList.isEmpty()) {
            return noGroupKeyInvokerList;
        }
        // 真不行，什么也找不到，随便转发
        return invokers;
    }

    private boolean groupKeyEquals(String requestGroupKey, String serverGroupKey, String targetNodeIp) {
        if (StringUtils.isBlank(requestGroupKey)) {
            requestGroupKey = StringUtils.EMPTY;
        }
        if (StringUtils.isBlank(serverGroupKey)) {
            serverGroupKey = StringUtils.EMPTY;
        }
        if (StringUtils.equals(requestGroupKey, serverGroupKey)) {
            return true;
        }
        if (requestGroupKey.startsWith(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX)) {
            String ipFromRequest = requestGroupKey.substring(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX.length());
            return ipFromRequest.equals(targetNodeIp);
        }
        return false;
    }
}
