package cn.com.duibaboot.ext.autoconfigure.cloud.zipkin;

import brave.Span;
import brave.Tracer;
import brave.propagation.ExtraFieldPropagation;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.sleuth.instrument.web.SleuthWebProperties;
import org.springframework.cloud.sleuth.instrument.web.TraceWebServletAutoConfiguration;
import org.springframework.context.ApplicationContext;
import org.springframework.core.annotation.Order;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Optional;
import java.util.regex.Pattern;

@Order(TraceWebServletAutoConfiguration.TRACING_FILTER_ORDER+1)
public class DuibaTraceSecondFilter implements Filter {

    private Pattern skipPattern = Pattern.compile(SleuthWebProperties.DEFAULT_SKIP_PATTERN);

    private static ApplicationContext applicationContext;

    private static volatile Tracer tracer;

    private static Tracer getTracer(){
        if (tracer == null) {
            tracer = applicationContext.getBean(Tracer.class);
        }
        return tracer;
    }

    @Autowired
    public void setApplicationContext(ApplicationContext applicationContext) {
        DuibaTraceSecondFilter.applicationContext = applicationContext;//NOSONAR
    }

    @Override
    public void init(FilterConfig filterConfig){
        //do nothing
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain chain) throws IOException, ServletException {
        if (!(servletRequest instanceof HttpServletRequest) || !(servletResponse instanceof HttpServletResponse)) {
            throw new ServletException("Filter just supports HTTP requests");
        }

        HttpServletRequest httpRequest = (HttpServletRequest) servletRequest;
        HttpServletResponse response = (HttpServletResponse) servletResponse;
        boolean skip = skipPattern.matcher(httpRequest.getRequestURI()).matches();
        if (!skip) {
            Span span = getTracer().currentSpan();
            String bagFlages = ExtraFieldPropagation.get(span.context(), ZipkinConstants.SPAN_FLAGS);
            String flages = (String) Optional.ofNullable(servletRequest.getAttribute(ZipkinConstants.SPAN_FLAGS)).orElse(ZipkinConstants.SPAN_NOT_SAMPLED);
            if(StringUtils.equalsIgnoreCase(ZipkinConstants.SPAN_SAMPLED,flages)||StringUtils.equalsIgnoreCase(ZipkinConstants.SPAN_SAMPLED,bagFlages)){
                ExtraFieldPropagation.set(span.context(), ZipkinConstants.SPAN_FLAGS, ZipkinConstants.SPAN_SAMPLED);
                response.setHeader(ZipkinConstants.TRACE_ID_NAME,span.context().traceIdString());
            }
        }
        chain.doFilter(servletRequest, servletResponse);
    }

    @Override
    public void destroy() {
        //do nothing
    }

    protected static Boolean isDebugMode(){
        if(applicationContext == null) {//applicationContext为null，说明还没启动好，直接返回false
            return false;
        }
        if(tracer==null || tracer.currentSpan() == null){//这里直接拿tracer而不是getTracer()，以防止StackOverflowError
            return false;
        }
        Span span = tracer.currentSpan();
        String bagFlages = ExtraFieldPropagation.get(span.context(), ZipkinConstants.SPAN_FLAGS);
        return StringUtils.equalsIgnoreCase(ZipkinConstants.SPAN_SAMPLED,bagFlages);
    }

}
