package cn.com.duibaboot.ext.autoconfigure.grouping.dubbo;

import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.rpc.Invocation;
import org.apache.dubbo.rpc.Invoker;
import org.apache.dubbo.rpc.RpcException;
import org.apache.dubbo.rpc.cluster.router.AbstractRouter;

import java.util.ArrayList;
import java.util.List;

/**
 * 用于服务分组调用的 dubbo路由器
 * 优先调用有相同【服务分组key（多场景）】的服务，如果找不到，那么调用【没有分组key（主场景）】的服务，如果还是没有，那么随便找一台调用
 * Created by gyf .
 * 2020/11/12 .
 */
public class ServiceGroupDubboRouter extends AbstractRouter {

    public ServiceGroupDubboRouter(URL url) {
        this.url = url;
    }

    @Override
    public <T> List<Invoker<T>> route(List<Invoker<T>> invokers, URL url, Invocation invocation) throws RpcException {
        if (CollectionUtils.isEmpty(invokers)) {
            return invokers;
        }

        String currentServiceGroupKey = ServiceGroupContext.getCurrentGroupKey();

        List<Invoker<T>> havePriorityList = new ArrayList<>(invokers.size());  // 优先转发目标服务列表
        List<Invoker<T>> noGroupKeyInvokerList = new ArrayList<>(invokers.size());  // 没有分组标识的服务
        for (Invoker<T> invoker : invokers) {
            String invokerGroupKey = invoker.getUrl().getParameter(ServiceGroupDubboRegistryFactoryWrapper.SERVICE_GROUP_KEY);
            String runInDocker = invoker.getUrl().getParameter(ServiceGroupDubboRegistryFactoryWrapper.RUN_IN_DOCKER);
            String targetNodeIp = invoker.getUrl().getIp();

            if (this.isPriorityGroup(currentServiceGroupKey, invokerGroupKey, targetNodeIp)) {
                havePriorityList.add(invoker);
            }

            if (this.isNoGroup(invokerGroupKey, runInDocker)) {
                // noGroupKeyServerList 满足以下所有场景
                // 1.主场景的服务
                // 2.运行在容器中的所有服务(本地服务属于在非容器中运行)
                noGroupKeyInvokerList.add(invoker);
            }
        }

        // 流量优先转发给具有相同分组标识的
        if (!havePriorityList.isEmpty()) {
            return havePriorityList;
        }

        // 1.currentServiceGroupKey不为空，但是没有找到优先转发的目标服务列表，使用noGroupKeyServerList
        // 2.currentServiceGroupKey为空，使用noGroupKeyServerList
        if (!noGroupKeyInvokerList.isEmpty()) {
            return noGroupKeyInvokerList;
        }

        // 真不行，什么也找不到，随便转发
        return invokers;
    }

    /**
     * 是否优先服务
     *
     * @param requestGroupKey
     * @param serverGroupKey
     * @param targetNodeIp
     * @return
     */
    private boolean isPriorityGroup(String requestGroupKey, String serverGroupKey, String targetNodeIp) {
        if (StringUtils.isBlank(requestGroupKey)) {
            return false;
        }
        // 本地调试优先调用本机服务
        if(requestGroupKey.startsWith(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX)){
            String ipFromRequest = requestGroupKey.substring(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX.length());
            return ipFromRequest.equals(targetNodeIp);
        }
        if (StringUtils.isBlank(serverGroupKey)) {
            return false;
        }
        // 只要key相等，就在优先投放列表
        return StringUtils.equals(requestGroupKey, serverGroupKey);
    }

    /**
     * 是否无分组服务
     *
     * @param serverGroupKey
     * @param runInDocker
     * @return
     */
    private boolean isNoGroup(String serverGroupKey, String runInDocker) {
        return StringUtils.isBlank(serverGroupKey) && "true".equals(runInDocker);
    }

}
