package cn.com.duibaboot.ext.autoconfigure.javaagent;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.JarURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.security.CodeSource;
import java.security.ProtectionDomain;
import java.util.Enumeration;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;

import cn.com.duibaboot.ext.autoconfigure.DuibaBootVersion;
import cn.com.duibaboot.ext.autoconfigure.core.utils.SpringBootUtils;
import com.ea.agentloader.AgentLoader;
import com.ea.agentloader.ClassPathUtils;
import net.bytebuddy.agent.ByteBuddyAgent;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.SpringApplicationRunListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.env.ConfigurableEnvironment;

/**
 * 应用启动后附加JavaAgent以修改类的字节码，进行一些aop处理
 */
public class AttachJavaAgentListener implements SpringApplicationRunListener {

    private static final Logger logger = LoggerFactory.getLogger(AttachJavaAgentListener.class);

    private static final String BOOT_INF_LIB_PREFIX = "BOOT-INF/lib/";
    private static final String BYTE_BYDDY_PREFIX = "byte-buddy";
    private static final String SPRING_BOOT_EXT_PREFIX_PREFIX = BOOT_INF_LIB_PREFIX + "spring-boot-ext-";

    public static final String DUIBA_NOAGENT = "duiba.noagent";

    private SpringApplication application;

    private static boolean startingCalled = false;

    /**
     * 必须有这个构造函数，否则spring无法初始化该类
     * @param application
     * @param args
     */
    public AttachJavaAgentListener(SpringApplication application, String[] args) {
        this.application = application;
    }

    //@Override //boot新版本没有这个方法，故去掉Override
    public void started() {
        this.starting();
    }

    // boot 1.5.2 新版本有这个方法，故要实现
    //@Override
    public void starting() {
        if(!startingCalled){
            startingCalled = true;//NOSONAR
            loadJavaAgent();
        }
    }

    private void loadJavaAgent(){
        if(noagent()){
            return;
        }
//        if(SpringBootUtils.isUnitTestMode()){
//            return;
//        }

        if(SpringBootUtils.isJarInJarMode()) {
            try {
                //把bytebuddy相关jar包导出到独立的jar包中，并添加进AppClassLoader的加载路径中。
                //如果不加上下面这行，则和自定义的SecurityManager会冲突，导致报错：java.lang.ClassCircularityError
                //TODO 后面去掉自定义的SecurityManager（改为bytebuddy实现）后，下面这行代码可以删除
                writeRequiredJar();

                //这个类需要在AppClassLoader中加载，需要导出到外部jar中。
//                String agentClass = "cn.com.duibaboot.ext.autoconfigure.javaagent.PluginAgentDelegate";
//                String jarPath = writeAgentClassesIntoJar(agentClass);
//                AgentLoader.loadAgent(jarPath, null);
            } catch (IOException | URISyntaxException e) {
                throw new RuntimeException(e);
            }
        }else{
            //如果是在IDEA/Eclipse/gradle命令行中运行，则直接加载即可
//            AgentLoader.loadAgentClass(PluginAgent.class.getName(), null);
        }

        //ByteBuddyAgent.install()内部做的其实是把net.bytebuddy.agent.Installer.class导出到外部jar中，使用此jar来启动attach，然后在Installer的静态变量中记录下instrumentation
        try {
            PluginAgent.agentmain(null, ByteBuddyAgent.install());
        }catch(IllegalStateException e){
            //failback，如果attach失败，可能是由于开发在使用jre开发，尝试换用AgentLoader来加载。AgentLoader会把jre缺失的attach相关类导出到一个jar中。
            try{
                AgentLoader.loadAgentClass(PluginAgent.class.getName(), null);
            }catch(Exception e1){
                //这个时候往logback打日志会被忽略，所以只能往标准错误流输出。
                System.err.println("调用attach注入javaagent失败，你是否使用了jre进行开发？请改为使用jdk");
                e.printStackTrace(System.err);
                throw e1;
            }
        }
    }

    /**
     * 启动脚本加入 -Dduiba.noagent=true 来避免在运行时加载javaagent
     * @return
     */
    private boolean noagent(){
        String noagent = System.getProperty(DUIBA_NOAGENT, "false");
        return "true".equals(noagent);
    }

    /**
     * 把agentClass这个类的class文件导出到一个独立的jar包中，包含 MANIFEST.MF 指定到这个class
     *
     * @param agentClass
     * @return
     * @throws IOException
     * @throws URISyntaxException
     */
    private String writeAgentClassesIntoJar(String agentClass) throws IOException, URISyntaxException {
        final File jarFile = File.createTempFile(".javaagent/javaagent." + agentClass, ".jar");
        jarFile.deleteOnExit();
        createAgentJar(new FileOutputStream(jarFile),
                agentClass,
                null,
                true,
                true,
                false);

        return jarFile.toString();
    }

    private void createAgentJar(
            final OutputStream out,
            final String agentClass,
            final String bootClassPath,
            final boolean canRedefineClasses,
            final boolean canRetransformClasses,
            final boolean canSetNativeMethodPrefix) throws IOException, URISyntaxException {
        final Manifest man = new Manifest();
        man.getMainAttributes().put(Attributes.Name.MANIFEST_VERSION, "1.0");
        man.getMainAttributes().putValue("Agent-Class", agentClass);
        if (bootClassPath != null)
        {
            man.getMainAttributes().putValue("Boot-Class-Path", bootClassPath);
        }
        man.getMainAttributes().putValue("Can-Redefine-Classes", Boolean.toString(canRedefineClasses));
        man.getMainAttributes().putValue("Can-Retransform-Classes", Boolean.toString(canRetransformClasses));
        man.getMainAttributes().putValue("Can-Set-Native-Method-Prefix", Boolean.toString(canSetNativeMethodPrefix));
        final JarOutputStream jarOut = new JarOutputStream(out, man);

        appendAgentClassToJar(agentClass, jarOut);

        jarOut.flush();
        jarOut.close();
    }

    /**
     *
     * 把byte-buddy等必要的jar写到独立的jar中，并加入AppClassLoader的加载路径
     * @throws URISyntaxException
     * @throws IOException
     */
    private void writeRequiredJar() throws URISyntaxException, IOException {
        String rootJarUrlStr = getRootJarUrl();
        URL rootJarUrl = new URL(rootJarUrlStr);

        JarURLConnection conn = (JarURLConnection)rootJarUrl.openConnection();

        JarFile jarFile = conn.getJarFile();
        Enumeration<JarEntry> entrys = jarFile.entries();
        while(entrys.hasMoreElements()){
            JarEntry entry = entrys.nextElement();
            String entryName = entry.getName();
            //把byte-buddy和byte-buddy-agent两个jar包导出到外部并加入AppClassLoader的加载路径
            if(entryName.startsWith(BOOT_INF_LIB_PREFIX + BYTE_BYDDY_PREFIX)){
                String nestedUrl = rootJarUrlStr + entry.getName()+"!/";
                String fileName = entryName.substring(BOOT_INF_LIB_PREFIX.length(), entryName.length() - ".jar".length());
                fileName = ".javaagent/" + fileName;
                URL url = writeJar(nestedUrl, fileName);

                //添加到AppClassLoader的加载路径
                ClassPathUtils.appendToSystemPath(url);
            }
        }
    }

    //eg.:   jar:file:/Users/wenqi.huang/Documents/workspace-idea/springBootWebDemo/build/libs/springBootWebDemo-0.0.1-SNAPSHOT.jar!/
    private String getRootJarUrl() throws URISyntaxException {
        ProtectionDomain protectionDomain = getClass().getProtectionDomain();
        CodeSource codeSource = protectionDomain.getCodeSource();
        URI location = (codeSource == null ? null : codeSource.getLocation().toURI());
        String path = (location == null ? null : location.getSchemeSpecificPart());
        if (path == null) {
            throw new IllegalStateException("Unable to determine code source archive");
        }
        if(path.startsWith("file:")){
            path = path.substring("file:".length());
        }
        int idx = path.indexOf("!/");
        if(idx > 0){
            path = path.substring(0,idx);
        }

        File root = new File(path);
        if (!root.exists()) {
            throw new IllegalStateException(
                    "Unable to determine code source archive from " + root);
        }

        String file = root.toURI() + "!/";
        file = file.replace("file:////", "file://"); // Fix UNC paths
        file = file.replace("file:/", "jar:file:/");
        return file;
    }

    /**
     * 把nestedJar写出到独立的jar中,先尝试写到用户目录下并缓存起来，如果写入失败则写到临时目录中。
     * @param url
     * @param fileName
     * @return
     * @throws IOException
     */
    private URL writeJar(String url, String fileName) throws IOException{
        boolean isSnapShot = fileName.endsWith("-SNAPSHOT");//如果jar名以-SNAPSHOT结尾的表示这个包可能随时变化，不允许重用
        URLConnection connection = new URL(url).openConnection();
        String userHome = getUserHome();
        File file = null;
        if(!isSnapShot) {
            file = new File(userHome + fileName + ".jar");
            if (file.exists()) {
                URL targetUrl = file.toURI().toURL();
                logger.info("jar file:{} exists, use it", targetUrl);
                return targetUrl;
            }
            file.getParentFile().mkdirs();
        }

        File tmpFile = null;
        if(!isSnapShot) {
            try {
                //先尝试在用户目录下建立jar文件，以便下次启动时重复使用（文件名随机，以防多个进程同时启动而写入冲突）
                tmpFile = File.createTempFile("tmp", "jar.tmp", file.getParentFile());
            } catch (IOException e) {
            }
        }
        if(tmpFile == null){
            //用户目录下创建文件失败 （或者是SNAPSHOT的情况），转而创建到临时目录下，临时目录下的文件只使用一次
            tmpFile = File.createTempFile(fileName, ".jar");
            tmpFile.deleteOnExit();
            file = null;
        }

        //遍历并写入到tmpFile
        if (connection instanceof JarURLConnection) {
            writeJarFileWithConnection((JarURLConnection) connection, tmpFile);
        }else{
            throw new IllegalStateException("[NOTIFYME] should not be here");
        }

        //如果是写入到用户目录下，则重命名为jar
        if(file != null){
            boolean success = tmpFile.renameTo(file);
            if(!success && !file.exists()){//文件重命名失败可能是其他进程已经重命名成功，所以这里重名失败后，判断文件是否存在，如果存在，则该文件可用
                throw new IllegalStateException("[NOTIFYME] should not be here");
            }
        }else{
            file = tmpFile;
        }

        URL targetUrl = file.toURI().toURL();
        logger.info("jar file:{} written completed", targetUrl);
        return targetUrl;
    }

    private void writeJarFileWithConnection(JarURLConnection connection, File tmpFile) throws IOException {
        JarFile jarFile = connection.getJarFile();
        byte[] bs = new byte[1024];

        try(JarOutputStream jarOut =
                    new JarOutputStream(new FileOutputStream(tmpFile))) {
            Enumeration<JarEntry> e = jarFile.entries();
            while (e.hasMoreElements()) {
                JarEntry jarEntry = e.nextElement();
                jarOut.putNextEntry(jarEntry);
                try (InputStream in = jarFile.getInputStream(jarEntry)) {
                    int n;
                    while ((n = in.read(bs)) != -1) {
                        jarOut.write(bs, 0, n);
                    }
                    jarOut.closeEntry();
                }
            }
//                jarOut.close();
        }
    }

    private String getUserHome(){
        String userHome = System.getProperty("user.home");
        if(!userHome.endsWith("/")){
            userHome = userHome + "/";
        }
        return userHome;
    }

    private void appendAgentClassToJar(String agentClass, JarOutputStream jarOut) throws IOException, URISyntaxException {
        String rootJarUrlStr = getRootJarUrl();
        //组装spring-boot-ext的jar url
        String nestedSpringBootExtUrlStr = rootJarUrlStr + SPRING_BOOT_EXT_PREFIX_PREFIX+ DuibaBootVersion.getVersion() + ".jar" + "!/";
        URL nestedSpringBootExtUrl = new URL(nestedSpringBootExtUrlStr);

        JarURLConnection conn = (JarURLConnection)nestedSpringBootExtUrl.openConnection();

        JarFile jarFile = conn.getJarFile();

        String agentClassentryName = agentClass.replaceAll("\\.", "/").concat(".class");

        JarEntry jarEntry = jarFile.getJarEntry(agentClassentryName);
        jarOut.putNextEntry(jarEntry);
        InputStream in = jarFile.getInputStream(jarEntry);
        IOUtils.copy(in, jarOut);
        jarOut.closeEntry();
    }

    //@Override
    public void environmentPrepared(ConfigurableEnvironment environment) {
        //do nothing
    }

    //@Override
    public void contextPrepared(ConfigurableApplicationContext context) {
        //do nothing
    }

    //@Override
    public void contextLoaded(ConfigurableApplicationContext context) {
        //do nothing
    }

    @Override
    public void started(ConfigurableApplicationContext context) {
        //do nothing
    }

    @Override
    public void running(ConfigurableApplicationContext context) {
        //do nothing
    }

    @Override
    public void failed(ConfigurableApplicationContext context, Throwable exception) {
        //do nothing
    }

}
