package cn.com.duibaboot.ext.autoconfigure.monitor.jvm;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duibaboot.ext.autoconfigure.dbexec.JvmIdTool;
import cn.com.duibaboot.ext.autoconfigure.monitor.jvm.alert.GcAlerter;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * gc监控
 * Created by guoyanfei .
 * 2021/6/8 .
 */
public class GcMonitor implements DisposableBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(GcMonitor.class);

    private static final String JSTAT_GC_COMMAND = "jstat -gc %s 5000";

    private volatile JstatThread jstatThread;

    @Autowired
    private List<GcAlerter> gcAlerters;

    @EventListener(MainContextRefreshedEvent.class)
    public void init() {
        jstatThread = new JstatThread();
        new Thread(jstatThread).start();
    }

    @Override
    public void destroy() throws Exception {
        jstatThread.stop();
    }

    private class JstatThread implements Runnable {

        /**
         * jstat线程取消状态，默认false
         */
        private volatile boolean jstatThreadCanceled = false;

        private final JstatAlerter jstatAlerter = new JstatAlerter();

        @Override
        public void run() {
            String command = String.format(JSTAT_GC_COMMAND, JvmIdTool.getVmId());
            ProcessBuilder processBuilder = new ProcessBuilder(command.split(" "));
            processBuilder.redirectErrorStream(true);
            Process p;
            try {
                p = processBuilder.start();
            } catch (IOException e) {
                LOGGER.warn("jstat命令启动异常, gc监控未启动, error={}", e.getMessage());
                return;
            }
            try (InputStream is = p.getInputStream();
                    BufferedReader bs = new BufferedReader(new InputStreamReader(is))) {
                String line;
                while ((line = bs.readLine()) != null) {
                    if (jstatThreadCanceled) {
                        break;
                    }
                    jstatAlerter.collect(line);
                }
            } catch (IOException e) {
                LOGGER.warn("jstat命令执行异常, gc监控停止, error={}", e.getMessage());
            }
            LOGGER.info("jstatThread completed!!");
        }

        public void stop() {
            jstatThreadCanceled = true;
        }
    }

    private class JstatAlerter {

        /**
         * 每5秒滚动一条jstat，1分钟就是12次
         */
        private static final int PERIOD_COUNT = 1;
//        private static final int PERIOD_COUNT = 12;

        /**
         * 统计队列最大长度60，表示一个小时
         */
        private static final int QUEUE_MAX_SIZE = 60;

        /**
         * 字段索引map
         * key: 字段
         * value: index，从0开始
         */
        private volatile Map<String, Integer> columnIndexMap;

        /**
         * 计数器，每5秒减1，
         */
        private final AtomicInteger periodCounter = new AtomicInteger(PERIOD_COUNT);

        private final LinkedList<JstatPoint> jstatPointQueue = new LinkedList<>();

        public void collect(String line) {
            // 正常情况下，line不会为空
            if (StringUtils.isBlank(line)) {
                return;
            }

            // 解析jstat每一行成一个数组
            String[] columns = line.trim().split(" +");

            // 当字段索引map为空的时候，说明是第一行(此处是单线程操作，没有并发问题)
            if (columnIndexMap == null) {
                Map<String, Integer> columnIndexMapTemp = new HashMap<>();
                for (int i = 0; i < columns.length; i++) {
                    columnIndexMapTemp.put(columns[i], i);
                }
                columnIndexMap = columnIndexMapTemp;
                return;
            }

            if (periodCounter.decrementAndGet() < 1) {
                String fgc = columns[columnIndexMap.get("FGC")];
                String fgct = columns[columnIndexMap.get("FGCT")];
                String ygc = columns[columnIndexMap.get("YGC")];
                String ygct = columns[columnIndexMap.get("YGCT")];
                jstatPointQueue.offer(new JstatPoint(fgc, fgct, ygc, ygct));
                if (jstatPointQueue.size() > QUEUE_MAX_SIZE) {
                    jstatPointQueue.poll();
                }
                tryAlert();
                // 重置计数器，下一分钟开始
                periodCounter.set(PERIOD_COUNT);
            }
        }

        /**
         * 尝试告警
         */
        public void tryAlert() {
            for (GcAlerter alerter : gcAlerters) {
                alerter.run(jstatPointQueue);
            }
        }

    }
}
