package cn.com.duibaboot.ext.autoconfigure.security.dpefensivepolicy;

import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.UUIDUtils;
import cn.com.duibaboot.ext.autoconfigure.security.DefensivePolicy;
import cn.com.duibaboot.ext.autoconfigure.security.exception.DuibaSecurityException;
import cn.com.duibaboot.ext.autoconfigure.web.login.LoginSuccessEvent;
import com.google.common.collect.Sets;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.multipart.MultipartResolver;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.servlet.DispatcherType;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.*;

/**
 * Csrf 防御策略
 */
public class CsrfDefensivePolicy extends HandlerInterceptorAdapter{

    private static final String CSRF_HEADER = "X-Csrf-Token";
    private static final String CSRF_COOKIE = "csrf_token";

    public static final String CSRF_PASS_MARK = "CSRF_PASS_MARK";

    @EventListener(LoginSuccessEvent.class)
    public void csrfCookieListening(LoginSuccessEvent event){
        String csrfToken = UUIDUtils.createSecureUUID();
        //不设置超时时间的是会话cookie,浏览器关闭即删除
        Cookie cookie = new Cookie(CSRF_COOKIE, csrfToken);
        cookie.setPath("/");
        cookie.setMaxAge(event.getExpirationTime());
        event.getResponse().addCookie(cookie);
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {

        if(!(handler instanceof HandlerMethod)){
            return true;
        }
        if (DispatcherType.REQUEST != request.getDispatcherType()) {//只处理REQUEST
            return true;
        }

        if(Optional.ofNullable(request.getAttribute(CSRF_PASS_MARK)).map((mark)-> (Boolean) mark).orElse(false)){
            return true;
        }
        HandlerMethod handlerMethod = (HandlerMethod)handler;

        String csrfTokenInCookie = getCookieValue(request);
        CsrfOff csrfOffAnno = handlerMethod.getMethodAnnotation(CsrfOff.class);
        if (!isAjaxRequest(handlerMethod) || csrfOffAnno != null || StringUtils.isBlank(csrfTokenInCookie)) {// 表示是ajax 请求
            return true;
        }
        String referer = request.getHeader("referer");
        if(!StringUtils.isEmpty(referer) && !referer.startsWith(RequestTool.getServerPath(request))){
            throw new DuibaSecurityException("invalid csrf token");
        }
        String csrfTokenInHeader = request.getHeader(CSRF_HEADER);
        if(!StringUtils.equals(csrfTokenInCookie,csrfTokenInHeader)){
            throw new DuibaSecurityException("invalid csrf token");
        }
        return true;
    }

    /**
     * 判断是否ajax请求
     */
    private boolean isAjaxRequest(HandlerMethod handlerMethod){
        ResponseBody responseBodyAnno = handlerMethod.getMethodAnnotation(ResponseBody.class);
        RestController restAnno = handlerMethod.getBeanType().getAnnotation(RestController.class);
        return(responseBodyAnno != null || restAnno != null || handlerMethod.getMethod().getReturnType().equals(Void.TYPE));//ajax 请求
    }

    private List<String> getCookieValues(HttpServletRequest request){
        List<String> values = new ArrayList<>();
        Cookie[] cookies=request.getCookies();
        if(cookies!=null){
            for (Cookie c : cookies) {
                if (CsrfDefensivePolicy.CSRF_COOKIE.equals(c.getName())) {
                    String value = c.getValue();
                    values.add(value);
                }
            }
        }
        return values;
    }

    private String getCookieValue(HttpServletRequest request){
        List<String> values = getCookieValues(request);
        if(values.isEmpty()){
            return null;
        }
        return values.get(values.size() - 1);
    }

}
