package cn.com.duibaboot.ext.autoconfigure.security.dpefensivepolicy;

import cn.com.duibaboot.ext.autoconfigure.core.utils.FileUtils;
import cn.com.duibaboot.ext.autoconfigure.security.DefensivePolicy;
import cn.com.duibaboot.ext.autoconfigure.security.SecurityPolicyProperties;
import cn.com.duibaboot.ext.autoconfigure.security.exception.DuibaSecurityException;
import com.google.common.base.Splitter;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.multipart.MultipartResolver;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Iterator;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 1.文件类型上传限制
 */
public class FileUploudDefensivePolicy implements DefensivePolicy {

    @Autowired
    private SecurityPolicyProperties policyProperties;

    private Set<String> whiteSuffix = Sets.newHashSet(
            "jpg","jpeg","png","gif",
            "xls","xlsx","csv","pdf","apk",
            "doc","docx","txt"
            );

    @Autowired
    private MultipartResolver resolver;

    @PostConstruct
    public void init(){

        String whiteSuffixStr = policyProperties.getUpload().getWhiteSuffixs();
        if(StringUtils.isBlank(whiteSuffixStr)){
            return;
        }
        Splitter splitter = Splitter.on(",").omitEmptyStrings().trimResults();
        whiteSuffix.addAll(splitter.splitToList(whiteSuffixStr).stream().map(StringUtils::lowerCase).collect(Collectors.toSet()));
    }

    @Override
    public Set<MediaType> getMediaTypes() {
        return Sets.newHashSet(MediaType.MULTIPART_FORM_DATA);
    }

    @Override
    public void doDefensive(HttpServletRequest request, HttpServletResponse response) throws DuibaSecurityException {

        if(!resolver.isMultipart(request)){
            return;
        }
        MultipartHttpServletRequest multipartRequest = resolver.resolveMultipart(request);
        Iterator<String> iterator = multipartRequest.getFileNames();
        while (iterator.hasNext()){
            MultipartFile multipartFile = multipartRequest.getFile(iterator.next());

            String originalFilename = multipartFile.getOriginalFilename();
            if(StringUtils.isBlank(originalFilename)){
                continue;
            }
            String suffix = FileUtils.getFileSuffix(originalFilename,false);
            if(StringUtils.isNotBlank(suffix) && !whiteSuffix.contains(StringUtils.lowerCase(suffix))){
                throw new DuibaSecurityException("不允许上传"+suffix+"类型的文件， 如果确实需要上传，请添加duiba.security.upload.whiteSuffixs配置，比如你希望上传html和txt文件，可以写duiba.security.upload.whiteSuffixs=html,txt");
            }
        }
    }

    public Set<String> getWhiteSuffix() {
        return whiteSuffix;
    }

    public void setWhiteSuffix(Set<String> whiteSuffix) {
        this.whiteSuffix = whiteSuffix;
    }

}
