package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon;

import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupUtils;
import com.netflix.loadbalancer.Server;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.annotation.Order;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * ribbon列表过滤器，此类用于配合容器组的测试环境多场景并行测试需求，容器组起的一些服务在注册到eureka中时会福袋分组标识，当前服务在发起调用前需要判断如下逻辑：
 * <br/>
 * 1、如果请求的Cookie/Header中带上了 _duibaServiceGroupKey=bizKey-id 的分组标识，
 * 流量优先转发给具有相同分组标识的 如果没有 -> 转发给没有分组标识的服务 如果没有 -> 随便转发
 * 2、如果请求的Cookie/Header中没有带上 _duibaServiceGroupKey=bizKey-id 的分组标识，
 * 流量优先转发给没有分组标识的服务 如果没有 -> 随便转发
 * Created by guoyanfei .
 * 2018/11/7 .
 */
@Order(-1)
public class ServiceGroupRibbonServerListFilter implements RibbonServerListFilter {

    @Override
    public List<Server> filter(List<Server> serverList, Object loadBalancerKey) {
        if (CollectionUtils.isEmpty(serverList)) {
            return serverList;
        }
        for (Server s : serverList) {
            if (!(s instanceof DiscoveryEnabledServer)) {
                return serverList;
            }
        }

        //优先从loadBalancerKey中获取，如果取不到才才从ServiceGroupContext中取（reactive不应该用ServiceGroupContext）
        String currentServiceGroupKey = null;
        if (loadBalancerKey != null && loadBalancerKey instanceof Map) {
            Map<String, Object> loadBalancerInfo = (Map<String, Object>)loadBalancerKey;
            currentServiceGroupKey = (String)loadBalancerInfo.get(ServiceGroupUtils.DUIBA_SERVICE_GROUP_KEY);
        }

        if(StringUtils.isBlank(currentServiceGroupKey)) {
            currentServiceGroupKey = ServiceGroupContext.getCurrentGroupKey();
        }

        List<Server> havePriorityList = new ArrayList<>(serverList.size());  // 优先转发目标服务列表
        List<Server> noGroupKeyServerList = new ArrayList<>(serverList.size());  // 没有分组标识的服务
        for (Server server : serverList) {
            String serverGroupKey = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_SERVICE_GROUP_KEY);
            String runInDocker = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.RUN_IN_DOCKER);
            String targetNodeIp = ((DiscoveryEnabledServer) server).getInstanceInfo().getIPAddr();

            if (this.isPriorityGroup(currentServiceGroupKey, serverGroupKey, targetNodeIp)) {
                havePriorityList.add(server);
            }

            if (this.isNoGroup(serverGroupKey, runInDocker)) {
                // noGroupKeyServerList 满足以下所有场景
                // 1.主场景的服务
                // 2.运行在容器中的所有服务(本地服务属于在非容器中运行)
                noGroupKeyServerList.add(server);
            }
        }

        // 流量优先转发给具有相同分组标识的
        if (!havePriorityList.isEmpty()) {
            return havePriorityList;
        }

        // 1.currentServiceGroupKey不为空，但是没有找到优先转发的目标服务列表，使用noGroupKeyServerList
        // 2.currentServiceGroupKey为空，使用noGroupKeyServerList
        if (!noGroupKeyServerList.isEmpty()) {
            return noGroupKeyServerList;
        }

        // 真不行，什么也找不到，随便转发
        return serverList;
    }

    /**
     * 是否优先服务
     *
     * @param requestGroupKey
     * @param serverGroupKey
     * @param targetNodeIp
     * @return
     */
    private boolean isPriorityGroup(String requestGroupKey, String serverGroupKey, String targetNodeIp) {
        if (StringUtils.isBlank(requestGroupKey)) {
            return false;
        }
        // 本地调试优先调用本机服务
        if(requestGroupKey.startsWith(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX)){
            String ipFromRequest = requestGroupKey.substring(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX.length());
            return ipFromRequest.equals(targetNodeIp);
        }
        if (StringUtils.isBlank(serverGroupKey)) {
            return false;
        }
        // 只要key相等，就在优先投放列表
        return StringUtils.equals(requestGroupKey, serverGroupKey);
    }

    /**
     * 是否无分组服务
     *
     * @param serverGroupKey
     * @param runInDocker
     * @return
     */
    private boolean isNoGroup(String serverGroupKey, String runInDocker) {
        return StringUtils.isBlank(serverGroupKey) && "true".equals(runInDocker);
    }

}
