package org.springframework.cloud.openfeign;

import cn.com.duiba.boot.cat.CatWithArgs;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duibaboot.ext.autoconfigure.cat.context.CatConstants;
import cn.com.duibaboot.ext.autoconfigure.cat.context.CatContext;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.AbstractMessage;
import com.google.common.annotations.VisibleForTesting;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

/**
 * 拦截FeignClient的调用加入cat监控功能，以及拦截并注入当前调用的方法
 */
@Slf4j
public class CustomFeignClientFactoryBean extends FeignClientFactoryBean {

    //客户端应用名，CustomFeignClientsRegistrar.customFeignClientBeanDefinition会注入
    private String springApplicationName;

    @Override
    public Object getObject() throws Exception {
        //增加cat监控功能
        Class classType = super.getObjectType();
        Object feignClient = super.getObject();

        return proxyInstance(classType, feignClient);
    }

    private Object proxyInstance(Class classType, Object feignClient) {
        if (CatUtils.isCatClassExists()) {
            return Proxy.newProxyInstance(classType.getClassLoader(), new Class[] { classType }, new FeignClientInvocationHandlerWithCat(springApplicationName, feignClient, this.getName()));
        } else {
            return Proxy.newProxyInstance(classType.getClassLoader(), new Class[] { classType }, new FeignClientInvocationHandler(springApplicationName, feignClient, this.getName()));
        }
    }

    public void setSpringApplicationName(String springApplicationName) {
        this.springApplicationName = springApplicationName;
    }

    /**
     * {@link MethodInterceptor}
     */
    @VisibleForTesting
    public static class FeignClientInvocationHandler implements InvocationHandler {

        protected final String springApplicationName;
        protected final String serverName;
        protected final Object original;

        FeignClientInvocationHandler(String springApplicationName, Object original, String serverName) {
            this.springApplicationName = springApplicationName;
            this.original = original;
            this.serverName = serverName;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                    || "hashCode".equals(methodName)
                    || "toString".equals(methodName)
                    || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            RpcContext.getContext().setMethod(method);
            RpcContext.getContext().setInvokeArgs(args);
            RpcContext.getContext().setSourceServiceId(springApplicationName);
            RpcContext.getContext().setTargetServiceId(serverName);

            return invokeMethodInner(method, original, args);
        }

        protected Object invokeMethodInner(Method method, Object target, Object[] args) throws Throwable{
            try {
                return method.invoke(target, args);
            } catch (Throwable e) {
                if(e instanceof InvocationTargetException){
                    e = ((InvocationTargetException)e).getTargetException();
                }
                //通常这步拿出来的异常只会有以下2种：HystrixBadRequestException, FeignClientException
                if(e instanceof HystrixBadRequestException && e.getCause() != null && e.getCause() instanceof BizException){
                    e = e.getCause();//如果cause是BizException，则应该把BizException提取出来抛出
                }
                throw e;
            }
        }

    }

    /**
     * {@link MethodInterceptor}
     */
    @VisibleForTesting
    public static class FeignClientInvocationHandlerWithCat extends FeignClientInvocationHandler {

        FeignClientInvocationHandlerWithCat(String springApplicationName, Object original, String serverName) {
            super(springApplicationName, original, serverName);
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                    || "hashCode".equals(methodName)
                    || "toString".equals(methodName)
                    || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            RpcContext.getContext().setMethod(method);
            RpcContext.getContext().setInvokeArgs(args);
            RpcContext.getContext().setSourceServiceId(springApplicationName);
            RpcContext.getContext().setTargetServiceId(serverName);

            if(!CatUtils.isCatEnabled()){
                return invokeMethodInner(method, original, args);
            }

            return invokeWithCatTransaction(method, args);
        }

        private Object invokeWithCatTransaction(Method method, Object[] args) throws Throwable{
            String loggerName = resolveLoggerName(method, args);

            Transaction transaction = Cat.newTransaction(CatConstants.CROSS_CONSUMER, loggerName);

            try {
                CatContext context = new CatContext();
                setAttachment(context);

                Object result = invokeMethodInner(method, original, args);

                transaction.setStatus(Message.SUCCESS);
                return result;
            }catch(Throwable e){
                if(!(e instanceof BizException)) {
                    Cat.logError(e);//BizException是正常的业务异常，比如用户不存在，不应该记录到cat.
                    transaction.setStatus(e);
                }else{
                    //记录BizException
                    Event bizExceptionEvent = Cat.newEvent("BizException", String.format("BizException(message:%s,code:%s)(this exception will not trigger hystrix)",e.getMessage(),((BizException) e).getCode()));
                    completeEvent(bizExceptionEvent);

                    transaction.addChild(bizExceptionEvent);

                    transaction.setStatus(Message.SUCCESS);
                }
                throw e;
            }finally {
                Event crossAppEvent =  Cat.newEvent(CatConstants.CONSUMER_CALL_APP, springApplicationName);
                Event crossServerEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_SERVER, serverName);
                completeEvent(crossAppEvent);
                completeEvent(crossServerEvent);

                transaction.addChild(crossAppEvent);
                transaction.addChild(crossServerEvent);

                CatUtils.completeTransaction(transaction);
            }
        }

        private void completeEvent(Event event){
            AbstractMessage message = (AbstractMessage) event;
            message.setStatus(Event.SUCCESS);
            message.setCompleted(true);
        }

        private String resolveLoggerName(Method method, Object[] args){//NOSONAR
            StringBuilder loggerName = new StringBuilder(method.getDeclaringClass().getSimpleName())
                    .append(".").append(method.getName());

            //注解了CatWithArgs表示cat中的监控名需要附加参数，详情见CatWithArgs类的注释
            if(method.isAnnotationPresent(CatWithArgs.class)){
                CatWithArgs catWithArgs = method.getAnnotation(CatWithArgs.class);
                int[] argIndexes = catWithArgs.argIndexes();
                if(argIndexes == null || argIndexes.length == 0
                        || args == null || args.length == 0){
                    return loggerName.toString();
                }

                StringBuilder sb = new StringBuilder();
                for(int argIdx : argIndexes){
                    if(argIdx >= 0 && args.length > argIdx) {
                        Object obj = args[argIdx];
                        sb.append(obj == null ? "" : obj.toString()).append(",");
                    }
                }

                if(sb.length() > 0){
                    sb.deleteCharAt(sb.length() - 1);
                    sb.insert(0, "(");
                    sb.append(")");
                }
                loggerName.append(sb);
            }
            return loggerName.toString();
        }

        private void setAttachment(CatContext context){
            RpcContext.getContext().setAttachment(Cat.Context.ROOT,context.getProperty(Cat.Context.ROOT));
            RpcContext.getContext().setAttachment(Cat.Context.CHILD,context.getProperty(Cat.Context.CHILD));
            RpcContext.getContext().setAttachment(Cat.Context.PARENT,context.getProperty(Cat.Context.PARENT));
            RpcContext.getContext().setAttachment(cn.com.duibaboot.ext.autoconfigure.core.Constants.X_RPC_CLIENT, springApplicationName);
        }

    }

}
