package cn.com.duibaboot.ext.autoconfigure.cat;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.wolf.cache.XMemcacheClient;
import cn.com.duiba.wolf.redis.RedisClient;
import cn.com.duibaboot.ext.autoconfigure.cat.annotation.CatTransactionAspect;
import cn.com.duibaboot.ext.autoconfigure.cat.context.CatInstance;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.CustomRequestInterceptor;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.duibaboot.ext.autoconfigure.threadpool.wrapper.ThreadPoolExecutorWrapper;
import com.dianping.cat.Cat;
import com.dianping.cat.servlet.CatFilter;
import com.dianping.cat.status.StatusExtension;
import com.dianping.cat.status.StatusExtensionRegister;
import io.lettuce.core.cluster.api.async.RedisClusterAsyncCommands;
import io.searchbox.client.JestClient;
import io.undertow.Undertow;
import net.rubyeye.xmemcached.MemcachedClient;
import org.apache.catalina.startup.Tomcat;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.common.extension.ExtensionLoader;
import org.apache.dubbo.common.store.DataStore;
import org.apache.dubbo.registry.RegistryService;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.http.client.HttpClient;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.session.SqlSessionFactory;
import org.aspectj.lang.annotation.Aspect;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.context.WebServerInitializedEvent;
import org.springframework.boot.web.embedded.tomcat.TomcatServletWebServerFactory;
import org.springframework.boot.web.embedded.tomcat.TomcatWebServer;
import org.springframework.boot.web.embedded.undertow.UndertowServletWebServer;
import org.springframework.boot.web.embedded.undertow.UndertowServletWebServerFactory;
import org.springframework.boot.web.server.WebServer;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.*;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.hadoop.hbase.HbaseTemplate;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.server.WebFilter;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import org.xnio.XnioWorker;
import redis.clients.jedis.Jedis;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.servlet.Servlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * Created by wenqi.huang on 2016/11/7.
 */
@Configuration
@EnableAspectJAutoProxy(proxyTargetClass = true)
@Import(SpringMvcConfigForCat.class)
@EnableConfigurationProperties(CatProperties.class)
public class CatAutoConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(CatAutoConfiguration.class);
    private static final String ThreadPoolWaterLevelId = "线程池水位图";

    @Resource
    private CatProperties catProperties;

    @EventListener({ EnvironmentChangeEvent.class, MainContextRefreshedEvent.class })
    @Order(Ordered.LOWEST_PRECEDENCE)
    public void refresh() {
        if (catProperties.isEnabled()) {
            CatInstance.enable();
        } else {
            CatInstance.disable();
        }
    }

    @Configuration
    @EnableConfigurationProperties(CatProperties.class)
    @ConditionalOnClass({Servlet.class, InterceptorRegistry.class})
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    @Conditional(CatCondition.class)
    @Order(-99)
    public static class CatHttpHandlerInterceptorServletConfiguration implements WebMvcConfigurer {

        @Resource
        private CatProperties catProperties;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            CatHandlerInterceptor chi = new CatHandlerInterceptor(catProperties) {

                @Override
                public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
                    if ("true".equals(request.getHeader(CustomRequestInterceptor.X_RPC))) {
                        //不处理RPC请求（来自FeignClient的rest rpc调用）
                        return true;
                    }

                    //如果不是前端资源，则继续cat的处理流程
                    return super.preHandle(request, response, handler);
                }

            };
            registry.addInterceptor(chi);
        }

    }

    @Configuration
    @ConditionalOnMissingClass("org.springframework.cloud.gateway.route.Route")//gateway中不生效
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.REACTIVE)
    @Conditional(CatCondition.class)
    public static class CatHttpHandlerInterceptorReactiveConfiguration {

        @Bean
        public WebFilter catWebFilter() {
            return new CatReactiveWebFilter();
        }

    }

    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({SqlSessionTemplate.class, SqlSessionFactoryBean.class, SqlSessionFactory.class,CatMybatisPlugin.class, Cat.class})
    public static class CatMyBatisPostProcessorConfiguration{

        /**
         * 声明后置处理器，spring全部bean初始化完成后调用，给所有SqlSessionBean注入CatMybatisPlugin plugin，监控sql的执行
         * @return
         */
        @Bean
        public static SpecifiedBeanPostProcessor catMyBatisPostProcessorConfigurer(){
            return new CatMybatisBeanPostProcessor();
        }
    }

    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({CloseableHttpAsyncClient.class, Cat.class})
    public static class CatHttpAsyncClientPostProcessorConfiguration{

        /**
         * 声明后置处理器，spring全部bean初始化完成后调用，让CloseableHttpAsyncClient支持Cat监控
         * @return
         */
        @Bean
        public static SpecifiedBeanPostProcessor catHttpAsyncClientPostProcessorConfigurer(){
            return new CatHttpAsyncClientPostProcessor();
        }
    }

    private static class CatMybatisBeanPostProcessor implements SpecifiedBeanPostProcessor<Object> {
        @Override
        public int getOrder() {
            return -1;
        }

        @Override
        public Class<Object> getBeanType() {
            return Object.class;
        }

        @Override
        public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
            return bean;
        }

        @Override
        public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
            SqlSessionFactory s = null;
            if(bean instanceof SqlSessionFactory){
                s = (SqlSessionFactory)bean;
            }
            if(bean instanceof SqlSessionTemplate){
                s = ((SqlSessionTemplate)bean).getSqlSessionFactory();
            }
            if(s == null){
                return bean;
            }

            addInterceptor(s);

            return bean;
        }

        private void addInterceptor(SqlSessionFactory s){
            boolean hasCatPlugin = false;
            if(!CollectionUtils.isEmpty(s.getConfiguration().getInterceptors())) {
                for (Interceptor plugin : s.getConfiguration().getInterceptors()) {
                    if (plugin instanceof CatMybatisPlugin) {
                        hasCatPlugin = true;
                        break;
                    }
                }
            }

            if (!hasCatPlugin) {
                s.getConfiguration().addInterceptor(new CatMybatisPlugin());
            }
        }
    }

    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({RedisClient.class, Jedis.class, Cat.class, Aspect.class})
    public static class CatRediscacheConfiguration{

    	@Bean
    	public CatRediscachePlugin getCatCachePlugin(){
    		return new CatRediscachePlugin();
    	}

    }

    /**
     * 加入aop，监控spring-data-redis执行耗时
     */
    @Configuration
    @ConditionalOnClass({JedisConnectionFactory.class, Jedis.class, Cat.class, Aspect.class})
    @Conditional(CatCondition.class)
    public static class CatSpringDataRedisAspectConfiguration{

    	@Bean
    	public CatSpringDataRedisPlugin catSpringDataRedisAspectPlugin(){
    		return new CatSpringDataRedisPlugin();
    	}

    }

    /**
     * 加入aop，监控spring-data-redis执行耗时
     */
    @Configuration
    @ConditionalOnClass({JedisConnectionFactory.class, RedisClusterAsyncCommands.class, Cat.class, Aspect.class})
    @Conditional(CatCondition.class)
    public static class CatSpringDataLettuceAspectConfiguration{

    	@Bean
    	public CatSpringDataLettucePlugin catSpringDataLettuceAspectPlugin(){
    		return new CatSpringDataLettucePlugin();
    	}

    }

    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({XMemcacheClient.class, MemcachedClient.class, Cat.class, Aspect.class})
    public static class CatMemcacheConfiguration{

    	@Bean
    	public CatMemcachePlugin getCatMemcachePlugin(){
    		return new CatMemcachePlugin();
    	}

    }

    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({MongoOperations.class, Cat.class, Aspect.class})
    public static class CatMongodbConfiguration{

    	@Bean
    	public CatSpringDataMongodbPlugin getCatSpringDataMongodbPlugin(){
    		return new CatSpringDataMongodbPlugin();
    	}

    }

    @Bean
    @Conditional(CatCondition.class)
    public CatTransactionAspect catTransactionAspect(){
        return new CatTransactionAspect();
    }

    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({ElasticsearchOperations.class, Cat.class, Aspect.class})
    public static class CatElasticsearchConfiguration{

    	@Bean
    	public CatSpringDataElasticSearchPlugin getCatSpringDataElasticSearchPlugin(){
    		return new CatSpringDataElasticSearchPlugin();
    	}

    }

    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({JestClient.class, Cat.class, Aspect.class})
    public static class CatJestElasticsearchConfiguration{

    	@Bean
    	public CatJestElasticSearchPlugin catJestElasticSearchPlugin(){
    		return new CatJestElasticSearchPlugin();
    	}

    }

    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({HbaseTemplate.class, HTable.class, Aspect.class})
    public static class CatHbaseConfiguration{

        @Bean
        public CatSpringDataHbaseAspect catSpringDataHbaseAspect(){
            return new CatSpringDataHbaseAspect();
        }

    }

    /**
     * httpClient cat 统计功能
     * @return
     */
    @Conditional(CatCondition.class)
    @ConditionalOnClass(HttpClient.class)
    @Configuration
    public static class CatHttpClientConfiguration{
        @Bean
        public CatHttpClientMethodInterceptor catHttpClientMethodInterceptor(){
            return new CatHttpClientMethodInterceptor();
        }
    }

    @Configuration
    @ConditionalOnClass({Cat.class})
    public static class InitCatServer implements ApplicationListener<ContextRefreshedEvent>, Ordered{

		@Override
		public void onApplicationEvent(ContextRefreshedEvent event) {
			//初始化cat监控
            //其实在CatCondition中已经初始化了cat，这里不是必要的
			CatInstance.isEnable();
		}

        @Override
        public int getOrder() {
            return -3;
        }
    }

    /**
     * 监控dubbo线程池使用情况，给cat发送心跳来记录
     */
    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({RegistryService.class, DataStore.class, CatFilter.class})
    public static class DubboThreadPoolWaterLevelHeartbeatExtenstion implements StatusExtension {

        //给当前的类定义一个extension id, cat服务端会将一个extension id作为一个group，然后将每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示
        @Override
        public String getId() {
            return ThreadPoolWaterLevelId;
        }

        //一个简单描述
        @Override
        public String getDescription() {
            return "各个线程池的水位监控心跳";
        }

        //每隔一分钟自动被cat调用一次
        @Override
        public Map<String, String> getProperties() {
            Map<String, String> maps = new HashMap<>();

            DataStore dataStore = ExtensionLoader.getExtensionLoader(DataStore.class).getDefaultExtension();
            Map<String, Object> executors = dataStore.get(CommonConstants.EXECUTOR_SERVICE_COMPONENT_KEY);

            for(Map.Entry<String, Object> entry : executors.entrySet()) {
                String port = entry.getKey();
                ExecutorService executor = (ExecutorService) entry.getValue();

                if (executor != null && executor instanceof ThreadPoolExecutor) {
                    ThreadPoolExecutor tp = (ThreadPoolExecutor) executor;

                    //这里的每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示,注意，只有value值可以被转化double类型的才会在heartbeat做图形展示
                    String portStr = executors.size() == 1 ? "" : ("port:" + port) + ",";
                    maps.put("Dubbo-ActiveCount("+portStr+"core:"+tp.getCorePoolSize()+",max:"+tp.getMaximumPoolSize()+")", String.valueOf(tp.getActiveCount()));
                }
            }

            return maps;
        }

        //这里是实现了初始化方法，把这个实现注册到cat上，如果你使用spring，需要在spring里面注册此bean，并实现初始化方法。
        @PostConstruct
        public void initialize() {
            StatusExtensionRegister.getInstance().register(this);
        }

    }

    /**
     * 监控Tomcat http线程池使用情况，给cat发送心跳来记录
     */
    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    @ConditionalOnClass({TomcatServletWebServerFactory.class, Tomcat.class, Servlet.class, CatFilter.class})
    public static class HttpThreadPoolWaterLevelHeartbeatExtenstion implements StatusExtension, ApplicationListener<WebServerInitializedEvent> {

        private ThreadPoolExecutor httpThreadPool;

        //给当前的类定义一个extension id, cat服务端会将一个extension id作为一个group，然后将每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示
        @Override
        public String getId() {
            return ThreadPoolWaterLevelId;
        }

        //一个简单描述
        @Override
        public String getDescription() {
            return "各个线程池的水位监控心跳";
        }

        //每隔一分钟自动被cat调用一次
        @Override
        public Map<String, String> getProperties() {
            Map<String, String> maps = new HashMap<>();

            if(httpThreadPool != null) {
                maps.put("Tomcat-ActiveCount(core:" + httpThreadPool.getCorePoolSize() + ",max:" + httpThreadPool.getMaximumPoolSize() + ")", String.valueOf(httpThreadPool.getActiveCount()));
            }

            return maps;
        }

        //这里是实现了初始化方法，把这个实现注册到cat上，如果你使用spring，需要在spring里面注册此bean，并实现初始化方法。
        @PostConstruct
        public void initialize() {
            //do nothing
        }

        @Override
        public void onApplicationEvent(WebServerInitializedEvent event) {
            WebServer c = event.getWebServer();

            if(!(c instanceof TomcatWebServer)){
                return;
            }
            TomcatWebServer container = (TomcatWebServer) c;

            Executor es = container.getTomcat().getConnector().getProtocolHandler().getExecutor();
            if(!(es instanceof ThreadPoolExecutor)) {
                return;
            }
            httpThreadPool = (ThreadPoolExecutor)es;
            //基本的校验通过才注册到cat回调
            StatusExtensionRegister.getInstance().register(this);
        }
    }

    /**
     * 监控Undertow http线程池使用情况，给cat发送心跳来记录
     */
    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    @ConditionalOnClass({UndertowServletWebServerFactory.class, Undertow.class, Servlet.class, CatFilter.class})
    public static class HttpThreadPoolWaterLevelHeartbeatExtenstionForUndertow implements StatusExtension, ApplicationListener<WebServerInitializedEvent> {

        private ThreadPoolExecutor httpThreadPool;

        //给当前的类定义一个extension id, cat服务端会将一个extension id作为一个group，然后将每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示
        @Override
        public String getId() {
            return ThreadPoolWaterLevelId;
        }

        //一个简单描述
        @Override
        public String getDescription() {
            return "各个线程池的水位监控心跳";
        }

        //每隔一分钟自动被cat调用一次
        @Override
        public Map<String, String> getProperties() {
            Map<String, String> maps = new HashMap<>();

            if(httpThreadPool != null) {
                maps.put("Undertow-ActiveCount(core:" + httpThreadPool.getCorePoolSize() + ",max:" + httpThreadPool.getMaximumPoolSize() + ")", String.valueOf(httpThreadPool.getActiveCount()));
            }

            return maps;
        }

        //这里是实现了初始化方法，把这个实现注册到cat上，如果你使用spring，需要在spring里面注册此bean，并实现初始化方法。
        @PostConstruct
        public void initialize() {
            //do nothing
        }

        @Override
        public void onApplicationEvent(WebServerInitializedEvent event) {
            WebServer c = event.getWebServer();

            if(!(c instanceof UndertowServletWebServer)){
                return;
            }
            UndertowServletWebServer container = (UndertowServletWebServer)c;

            Field field = ReflectionUtils.findField(container.getClass(), "undertow");
            field.setAccessible(true);
            Undertow undertow = (Undertow)ReflectionUtils.getField(field, container);

            XnioWorker worker = undertow.getWorker();
            field = ReflectionUtils.findField(worker.getClass(), "taskPool");
            field.setAccessible(true);
            ThreadPoolExecutor threadPoolExecutor = (ThreadPoolExecutor)ReflectionUtils.getField(field, worker);
            this.httpThreadPool = threadPoolExecutor;

            //基本的校验通过才注册到cat回调
            StatusExtensionRegister.getInstance().register(this);
        }
    }

    /**
     * 监控统一线程池使用情况，给cat发送心跳来记录
     */
    @Configuration
    @Conditional(CatCondition.class)
    @ConditionalOnClass({CatFilter.class})
    public static class DubiaThreadPoolWaterLevelHeartbeatExtenstion implements StatusExtension, ApplicationListener<ContextRefreshedEvent> {

        private List<ThreadPoolExecutorWrapper> threadPools;

        //给当前的类定义一个extension id, cat服务端会将一个extension id作为一个group，然后将每个key指标都会做图形处理，结果会在cat的heartbeat中进行展示
        @Override
        public String getId() {
            return ThreadPoolWaterLevelId;
        }

        //一个简单描述
        @Override
        public String getDescription() {
            return "各个线程池的水位监控心跳";
        }

        //每隔一分钟自动被cat调用一次
        @Override
        public Map<String, String> getProperties() {
            Map<String, String> maps = new HashMap<>();
            if(threadPools != null){
                for(ThreadPoolExecutorWrapper p : threadPools) {
                    maps.put(p.getThreadPoolName() + "-ActiveCount(core:" + p.getCorePoolSize() + ",max:" + p.getMaximumPoolSize() + ")", String.valueOf(p.getActiveCount()));
                }
            }

            return maps;
        }

        //这里是实现了初始化方法，把这个实现注册到cat上，如果你使用spring，需要在spring里面注册此bean，并实现初始化方法。
        @PostConstruct
        public void initialize() {
            //do nothing
        }

        @Override
        public void onApplicationEvent(ContextRefreshedEvent event) {
            ApplicationContext applicationContext = event.getApplicationContext();

            try {
                Map<String, ThreadPoolExecutorWrapper> beansMap = applicationContext.getBeansOfType(ThreadPoolExecutorWrapper.class);
                if(beansMap != null && !beansMap.isEmpty()){
                    this.threadPools = new ArrayList<>(beansMap.values());
                }
            }catch(Exception e){
                //Ignore
            }

            if(this.threadPools != null){
                StatusExtensionRegister.getInstance().register(this);
            }
        }
    }

    private static class CatCondition implements Condition{

        private static volatile boolean initted = false;

        @Override
        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            checkCatInited(context.getEnvironment());

            //这个方法内部会延迟初始化cat
            return CatUtils.isCatEnabled();
        }

        private synchronized void checkCatInited(Environment env){
            if(initted){
                return;
            }
            initted = true;

            String appName = env.getProperty("spring.application.name");
            String catServerIps = env.getProperty("cat.server.ips");

            if (StringUtils.isEmpty(appName)) {
                logger.warn("you need to set property[spring.application.name] in bootstrap.properties");
                return;
            }
            System.setProperty("cat.app.name", appName);//注入appName,以方便后续cat获取

            for(String p : env.getActiveProfiles()){
                if(SpringEnvironmentUtils.DEV.equals(p)){
                    String userHome = System.getProperty("user.home");
                    String logPath = userHome + File.separator + "logs" + File.separator + "duibacat" + File.separator;
                    System.setProperty("CAT_HOME", logPath);//注入cat写日志文件目录,避免写在默认根目录无权限
                }
            }

            if(!StringUtils.isBlank(catServerIps) && CatUtils.isCatClassExists()) {
                System.setProperty("cat.server.ips", catServerIps);//注入cat服务器ip地址列表(多个用逗号分隔),以方便后续cat获取,使用这个以后不再需要在客户端机器上新建cat.xml

                //在这之前cat可能过早地被初始化了，但是因为找不到服务器配置而没有初始化成功，这里判断如果没有初始化成功则销毁并重新初始化。
                if (!CatUtils.isCatEnabled()) {
                    //Field field = ReflectionUtils.findField(Cat.class, "s_init");
                    //field.setAccessible(true);
                    //ReflectionUtils.setField(field, null, false);

                    //ContainerLoader.destroyDefaultContainer();
                }
            }
        }
    }

}
