package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import cn.com.duiba.wolf.utils.ConcurrentUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer.IpAffinityServerListFilter;
import com.netflix.loadbalancer.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.web.reactive.context.ReactiveWebApplicationContext;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.cloud.netflix.ribbon.RibbonApplicationContextInitializer;
import org.springframework.cloud.netflix.ribbon.SpringClientFactory;
import org.springframework.cloud.openfeign.CustomFeignClientsRegistrar;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.context.WebApplicationContext;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 在 RibbonClientConfiguration 执行之前先定义自己的一些ribbon组件，以达到如下目的：<br/>
 * 1.使用自定义的ping组件，每隔3秒访问服务器/monitor/check接口，如果不通或者返回不是OK，则标记为dead状态，去除对该服务器的流量（官方的只判断了从eureka取下来的状态是否UP，存在很大延迟.）<br/>
 * 2.使用自定义的支持预热的rule组件（rule组件用于在每次调用接口时取得一个服务器），先尝试从isAlive的服务器中选一个，如果拿不到再转发到上层(上层方法默认获取全部服务器,并没有判断isAlive状态)<br/>
 * 3.设置ILoadBalancer中的pingStrategy，官方的pingStrategy是顺序执行的，假设服务器很多，执行速度会较慢(对于一次对全部服务器的ping，有个默认超时时间5秒)，这里替换成使用并发执行的策略,以加快速度
 */
@Configuration
@ConditionalOnClass({IPing.class, IRule.class, ILoadBalancer.class, IPingStrategy.class})
public class RibbonCustomAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(RibbonCustomAutoConfiguration.class);

    @Resource
    private SpringClientFactory springClientFactory;

    @Autowired
    private ObjectProvider<List<RibbonApplicationContextInitializer>> ribbonApplicationContextInitializersProvider;

    /**
     * 提前初始化ribbonContext，按照spring默认处理策略，spring会在ApplicationReadyEvent事件发生后初始化，此时已经太晚，当前实例已经注册到eureka上，此方法的作用就是确保注册到eureka之前已经把ribbon context初始化完毕
     */
    @EventListener(MainContextRefreshedEvent.class)
    @Order(Ordered.HIGHEST_PRECEDENCE) // 最高优先级
    public void onMainContextRefreshed(){
        long start = System.currentTimeMillis();
        List<RibbonApplicationContextInitializer> list = ribbonApplicationContextInitializersProvider.getIfAvailable();
        if(list != null){
            list.forEach(ribbonApplicationContextInitializer -> {
                ribbonApplicationContextInitializer.onApplicationEvent(null);
            });
        }
        long cost = System.currentTimeMillis() - start;
        logger.info("ribbon context initted in {}ms", cost);
    }

    /**
     * 扫描所有Feign客户端名，并初始化ribbon。以防止启动后第一次进行Feign调用很慢的问题。
     * @return
     */
    @Bean
    public ApplicationListener<ContextRefreshedEvent> springClientFactoryInitListener(){
        return new ApplicationListener<ContextRefreshedEvent>() {
            @Override
            public void onApplicationEvent(ContextRefreshedEvent event) {
                //扫描所有FeignClient并初始化！！每个大约要两秒，所以要并发运行
                if(event.getApplicationContext() instanceof WebApplicationContext
                        || event.getApplicationContext() instanceof ReactiveWebApplicationContext) {//只在主ApplicationContext中运行，以防止死循环
                    Set<String> enabledFeignClientNames = CustomFeignClientsRegistrar.getEnabledFeignClientNames();
//                    if(enabledFeignClientNames.size() <= 2){
                    for(final String name : enabledFeignClientNames) {
                        springClientFactory.getClientConfig(name);
                    }
//                    }
//                    else if(enabledFeignClientNames.size() > 1) {
//                        ExecutorService executor = Executors.newCachedThreadPool();
//                        List<Runnable> runnables = new ArrayList<>(enabledFeignClientNames.size());
//                        for(final String name : enabledFeignClientNames){
//                            runnables.add(() -> springClientFactory.getClientConfig(name));
//                        }
//
//                        try {
//                            ConcurrentUtils.executeTasksBlocking(executor, runnables);
//                        } catch (InterruptedException e) {
//                            //Ignore
//                        }finally{
//                            executor.shutdownNow();
//                        }
//                    }
                }
            }
        };
    }

    @Bean
    public IPingStrategy getConcurrentRibbonPingStrategy(){
        return new ConcurrentPingStrategy();
    }

    /**
     * 设置ILoadBalancer中的pingStrategy，官方的pingStrategy是顺序执行的，假设服务器很多，执行速度会较慢(对于一次对全部服务器的ping，有个默认超时时间5秒)，这里替换成使用并发执行的策略,以加快速度
     * @return
     */
    @Bean
    public ApplicationListener<ContextRefreshedEvent> iLoadBalancerInitListener(){
        //这里不用BeanPostProcessor是因为ILoadBalancer是在spring子容器中生成的，子容器默认不会应用BeanPostProcessor，但ApplicationListener仍然生效
        return new ApplicationListener<ContextRefreshedEvent>() {
            @Override
            public void onApplicationEvent(ContextRefreshedEvent event) {
                try {
                    ILoadBalancer bean = event.getApplicationContext().getBean(ILoadBalancer.class);
                    Field pingStrategyField = ReflectionUtils.findField(bean.getClass(), "pingStrategy");
                    if(pingStrategyField != null) {
                        pingStrategyField.setAccessible(true);
                        ReflectionUtils.setField(pingStrategyField, bean, getConcurrentRibbonPingStrategy());
                    }else{
                        logger.error("[NOTIFYME]ILoadBalancer中的pingStrategy Field不存在，无法设置使用并发ping策略");
                    }
                }catch(NoSuchBeanDefinitionException e){
                    //Ignore
                }
            }
        };
    }

    /**
     * ribbon列表过滤器,此类用于跟压测配合，用于隔离压测流量和正常流量，压测专用服务在注册到eureka中时会附带压测场景ID 和 独立容器集群压测标志，当前服务在发起调用前需要判读如下逻辑：
     * <br/>
     * 1. 如果当前请求带有场景ID && 独立容器集群压测，则过滤eureka服务列表的时候只会选择有相同场景ID的目标服务器进行调用
     * 2. 其余情况，则过滤eureka服务列表的时候只会选择没有场景ID的目标服务器进行调用
     *
     * <br/>
     * 这个类不能加到压测包下，因为压测包下的类只有在引入perftest模块的情况下才会生效，而此类在没有引入perftst模块的情况下也应该生效，否则正常流量可能会调用到压测专用服务器上。
     * <br/>
     */
    @Bean
    public PerfTestRibbonServerListFilter perfTestRibbonServerListFilter(){
        return new PerfTestRibbonServerListFilter();
    }

    /**
     * 判断如果有和当前IP一样的IP的服务（或者Ip前三个字节一样），则优先调度。策略默认关闭，需要通过配置显式启用
     */
    @Bean
    @RefreshScope
    public IpAffinityServerListFilter ipAffinityServerListFilter(){
        return new IpAffinityServerListFilter();
    }

    /**
     * ribbon列表过滤器，此类用于配合容器组的测试环境多场景并行测试需求，容器组起的一些服务在注册到eureka中时会福袋分组标识，当前服务在发起调用前需要判断如下逻辑：
     * <br/>
     * 1、如果请求的Cookie/Header中带上了 _duibaServiceGroupKey=bizKey-id 的分组标识，
     * 流量优先转发给具有相同分组标识的 如果没有 -> 转发给没有分组标识的服务 如果没有 -> 随便转发
     * 2、如果请求的Cookie/Header中没有带上 _duibaServiceGroupKey=bizKey-id 的分组标识，
     * 流量优先转发给没有分组标识的服务 如果没有 -> 随便转发
     * @return
     */
    @Bean
    public ServiceGroupRibbonServerListFilter serviceGroupRibbonServerListFilter() {
        return new ServiceGroupRibbonServerListFilter();
    }

    /**
     * 官方是顺序执行的，改为使用并发执行ping
     */
    private static class ConcurrentPingStrategy implements IPingStrategy, DisposableBean {
        private final ExecutorService executor = Executors.newFixedThreadPool(20, new NamedThreadFactory("ribbonPing"));

        private static final Logger logger = LoggerFactory.getLogger(ConcurrentPingStrategy.class);

        @Override
        public boolean[] pingServers(final IPing ping, Server[] servers) {
            int numCandidates = servers.length;
            boolean[] results = new boolean[numCandidates];/* Default answer is DEAD. */

            if (logger.isDebugEnabled()) {
                logger.debug("LoadBalancer:  PingTask executing ["
                        + numCandidates + "] servers configured");
            }
            if(ping == null){
                logger.error("IPing is null, will mark all servers as not alive");
                return results;
            }

            List<Callable<Boolean>> callables = new ArrayList<>();
            for (final Server server : servers) {
                callables.add(() -> ping.isAlive(server));
//                results[i] = ping.isAlive(servers[i]);
            }

            if(executor.isTerminated()){
//                logger.error("IPing's executor is terminated(most likely your ApplicationContext is destroyed), will mark all servers as not alive");
                return results;
            }

            try {
                List<Boolean> ret = ConcurrentUtils.submitTasksBlocking(executor, callables);
                int i = 0;
                for(Boolean b : ret){
                    results[i++] = b;
                }
            } catch (InterruptedException e) {
                //Ignore
                Thread.currentThread().interrupt();
            }

            return results;
        }

        @Override
        public void destroy() throws Exception {
            executor.shutdown();
        }
    }

}
