package cn.com.duibaboot.ext.autoconfigure.cat;

import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.DefaultTransaction;
import io.lettuce.core.RedisFuture;
import io.lettuce.core.cluster.api.async.RedisClusterAsyncCommands;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.data.redis.connection.RedisConnection;

/**
 * 加入aop，监控spring-data-redis执行耗时
 */
@Aspect
public class CatSpringDataLettucePlugin {

	private static final Logger logger = LoggerFactory.getLogger(CatSpringDataLettucePlugin.class);

	@Around("execution(* org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory.getConnection(..))")
	public Object springDataRedisJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable{
		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		String methodName = signature.getMethod().getName();
		if(CatUtils.isCatEnabled()){
			if("getConnection".equals(methodName)){
				try {
					RedisConnection connection = (RedisConnection) joinPoint.proceed();

					ProxyFactory factory = new ProxyFactory();
					factory.setTarget(connection);
					//factory.setTargetClass(JedisConnection.class);
					factory.addAdvice(new RedisConnectionMethodInterceptor());
					RedisConnection newConnection = (RedisConnection) factory.getProxy();//getClass().getClassLoader()

					return newConnection;
				} catch (Throwable e) {
					//失败才记录
					Transaction transaction = Cat.newTransaction("Cache.Lettuce", methodName);
					Cat.logError(e);
					transaction.setStatus(e);
					transaction.complete();
					throw e;
				}
			}
		}

		return joinPoint.proceed();
	}

	/**
	 * {@link MethodInterceptor}
	 */
	private static class RedisConnectionMethodInterceptor implements MethodInterceptor {

		RedisConnectionMethodInterceptor() {
		}

		@Override
		public Object invoke(MethodInvocation invocation) throws Throwable {
			String methodName = invocation.getMethod().getName();

			if(methodName.equals("isPipelined") || methodName.equals("openPipeline") || methodName.equals("isQueueing")
					|| methodName.equals("isClosed")
					|| methodName.equals("close")
					|| methodName.equals("closePipeline")
					|| methodName.endsWith("Commands")){
				return invocation.proceed();
			}else if(methodName.equals("getNativeConnection")){
				Object nativeConnection = invocation.proceed();

				Object newNativeConnection = nativeConnection;
				ProxyFactory factory = new ProxyFactory();
				factory.setTarget(nativeConnection);
				if(nativeConnection instanceof RedisClusterAsyncCommands){//异步
					factory.addAdvice(new LettuceAsyncMethodInterceptor());
					newNativeConnection = factory.getProxy();
				}

				return newNativeConnection;
			}

			return interceptorRedisMethods(invocation);
		}

	}

	/**
	 * {@link MethodInterceptor}
	 */
	private static class LettuceAsyncMethodInterceptor implements MethodInterceptor {

		@Override
		public Object invoke(MethodInvocation invocation) throws Throwable {	// NOSONAR
			if(!invocation.getMethod().getReturnType().equals(RedisFuture.class)) {
				return invocation.proceed();
			}
			String methodName = invocation.getMethod().getName();

			long startTime = System.nanoTime();

			//cat无法跨线程追踪，所以下面的redis记录在cat上不会形成调用链
			//这里算出来的rt有个小缺陷，如果用户端使用future.get(500,TimeUnit.MILLISECONDS)来调用，真实调用花了1秒，用户端会超时，而这里记录的不会超时
			RedisFuture future = (RedisFuture)invocation.proceed();
			future.whenComplete((s, throwable) -> {
				Transaction transaction;
				if("get".equals(methodName)){
					transaction = Cat.newTransaction("Cache.Lettuce", methodName + ":" + methodName);
				}else{
					transaction = Cat.newTransaction("Cache.Lettuce", methodName);
				}
				((DefaultTransaction)transaction).setDurationStart(startTime);//强制设置开始时间

				if(throwable != null){
					Cat.logError((Throwable)throwable);
					transaction.setStatus((Throwable)throwable);
				}
				else {
					if(s != null && "get".equals(methodName)) {
						Cat.logEvent("Cache.Lettuce", methodName + ":missed");
					}
					transaction.setStatus(Message.SUCCESS);
				}
				transaction.complete();
			});
			return future;
		}

	}

	private static Object interceptorRedisMethods(MethodInvocation invocation) throws Throwable {
		String methodName = invocation.getMethod().getName();
		//忽略Object基类中的方法
		if(methodName.equals("toString") || methodName.equals("hashCode") || methodName.equals("equals")){
			return invocation.proceed();
		}

		Transaction transaction;
		if("get".equals(methodName)){
			transaction = Cat.newTransaction("Cache.Lettuce", methodName + ":" + methodName);
		}else{
			transaction = Cat.newTransaction("Cache.Lettuce", methodName);
		}
		try {
			Object value = invocation.proceed();
			if(value == null && "get".equals(methodName)){
				Cat.logEvent("Cache.Lettuce", methodName + ":missed");
			}
			transaction.setStatus(Message.SUCCESS);
			return value;
		} catch (Throwable e) {
			Cat.logError(e);
			transaction.setStatus(e);
			throw e;
		}finally{
			transaction.complete();
		}
	}

}
