package cn.com.duibaboot.ext.autoconfigure.perftest.filter;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duiba.boot.perftest.PerfTestDto;
import cn.com.duiba.boot.perftest.ReactivePerfTestUtils;
import cn.com.duibaboot.ext.autoconfigure.core.utils.ReactiveHttpRequestUtils;
import cn.com.duibaboot.ext.autoconfigure.perftest.PerfTestFootMarker;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilter;
import org.springframework.web.server.WebFilterChain;
import reactor.core.publisher.Mono;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 压测用的Filter,reactive版,当从request识别到是压测请求时，设置attribute。
 * <br/>
 * 识别到当前服务器cpu高时直接拒绝当前压测请求，如果5秒内拒绝了10个以上压测请求则直接熔断5秒。
 */
@Slf4j
@Order(Ordered.HIGHEST_PRECEDENCE + 3)
public class ReactivePerfTestFilter implements WebFilter {

    @Resource
    private PerfTestRejectProperties perfTestRejectProperties;

    @Resource
    private PerfTestFootMarker perfTestFootMarker;

    private LoadingCache<String, RejectedCountDto> servicePerfTestRejectedRequestCountCache = Caffeine.newBuilder()
            .expireAfterWrite(5, TimeUnit.SECONDS)
            .initialCapacity(1)
            .maximumSize(1)
            .build(s -> new RejectedCountDto());

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, WebFilterChain chain) {
        boolean isPerfTestMode = ReactivePerfTestUtils.isPerfTestRequest(exchange.getRequest());

        if(isPerfTestMode) {
            String perfTestSceneId = ReactivePerfTestUtils.getPerfTestSceneId(exchange.getRequest());
            boolean isTestCluster = ReactivePerfTestUtils.isPerfTestCluster(exchange.getRequest());

            if (StringUtils.isNotBlank(perfTestSceneId)) {
                //约定ribbon的loadBalancerKey为Map<String, Object>
                Map<String, Object> loadBalancerInfo = exchange.getAttribute(RibbonServerListFilter.GATEWAY_LOAD_BALANCER_KEY_ATTR);
                if(loadBalancerInfo == null){
                    loadBalancerInfo = new HashMap<>();
                    exchange.getAttributes().put(RibbonServerListFilter.GATEWAY_LOAD_BALANCER_KEY_ATTR, loadBalancerInfo);
                }
                loadBalancerInfo.put(PerfTestDto.PERF_TEST_DTO_KEY, new PerfTestDto(perfTestSceneId, isTestCluster));
            }

//            perfTestFootMarker.markApp();//TODO markApp内部不能用Context（reactive的关系）
            if(isServerBusy()){
                exchange.getResponse().setStatusCode(HttpStatus.TOO_MANY_REQUESTS);//429 too many requests.
                return ReactiveHttpRequestUtils.write(exchange.getResponse(), "server busy(_duibaPerf=true)");//只有压测时才会显示这个信息
            }
        }

        return chain.filter(exchange);
    }

    private boolean isServerBusy(){
        //TODO 判断当前服务器cpu>90%直接拒绝压测请求
        return false;
    }

    /**
     * 判断压测请求是否熔断, 如果5秒内压测请求被拒绝的数量超过10个，则熔断5秒
     * @return
     */
    private boolean isCircuitBreak(){
        RejectedCountDto rejectedCountDto = servicePerfTestRejectedRequestCountCache.get("");

        if(rejectedCountDto.isCircuitBreak(perfTestRejectProperties.getCircuitBreakThreshold())){
            if(!rejectedCountDto.isLogged){
                rejectedCountDto.isLogged = true;

                //这个日志5秒最多打印一次，不会打印太多，以防影响性能
                log.warn("压测时发现servlet容器线程池过于繁忙，已熔断5秒内的所有压测请求");
            }
            return true;
        }

        return false;
    }

    /**
     * 添加压测请求拒绝数
     */
    private void addRejectCount(){
        RejectedCountDto rejectedCountDto = servicePerfTestRejectedRequestCountCache.get("");
        rejectedCountDto.incrRejectedCount();
    }

}
