package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupUtils;
import com.netflix.loadbalancer.Server;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.annotation.Order;

import java.util.ArrayList;
import java.util.List;

/**
 * ribbon列表过滤器，此类用于配合容器组的测试环境多场景并行测试需求，容器组起的一些服务在注册到eureka中时会福袋分组标识，当前服务在发起调用前需要判断如下逻辑：
 * <br/>
 * 1、如果请求的Cookie/Header中带上了 _duibaServiceGroupKey=bizKey-id 的分组标识，
 * 流量优先转发给具有相同分组标识的 如果没有 -> 转发给没有分组标识的服务 如果没有 -> 随便转发
 * 2、如果请求的Cookie/Header中没有带上 _duibaServiceGroupKey=bizKey-id 的分组标识，
 * 流量优先转发给没有分组标识的服务 如果没有 -> 随便转发
 * Created by guoyanfei .
 * 2018/11/7 .
 */
@Order(-1)
public class ServiceGroupRibbonServerListFilter implements RibbonServerListFilter {

    @Override
    public List<Server> filter(List<Server> serverList, Object loadBalancerKey) {
        if (CollectionUtils.isEmpty(serverList)) {
            return serverList;
        }
        for (Server s : serverList) {
            if (!(s instanceof DiscoveryEnabledServer)) {
                return serverList;
            }
        }

        String currentServiceGroupKey  = ServiceGroupUtils.getCurrentGroupKey();

        List<Server> havePriorityList = new ArrayList<>(serverList.size());  // 优先转发目标服务列表
        List<Server> noGroupKeyServerList = new ArrayList<>(serverList.size());  // 没有分组标识的服务
        for (Server server : serverList) {
            String serverGroupKey = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_SERVICE_GROUP_KEY);

            if (this.isPriorityGroup(currentServiceGroupKey, serverGroupKey)) {
                // priorityGroupServer：匹配到key的所有服务列表
                havePriorityList.add(server);
            }

            if (this.isNoGroup(serverGroupKey)) {
                // noGroupServer：所有serverGroupKey为空的server
                noGroupKeyServerList.add(server);
            }
        }

        // 流量优先转发给具有相同分组标识的
        if (!havePriorityList.isEmpty()) {
            return havePriorityList;
        }

        // 没有优先服务分组，且无分组服务不为空，就转发给无服务分组
        if (!noGroupKeyServerList.isEmpty()) {
            return noGroupKeyServerList;
        }

        // 真有问题的话，全部返回
        return serverList;
    }

    /**
     * 是否优先服务
     *
     * @param requestGroupKey
     * @param serverGroupKey
     * @return
     */
    private boolean isPriorityGroup(String requestGroupKey, String serverGroupKey) {
        if (StringUtils.isBlank(requestGroupKey) || StringUtils.isBlank(serverGroupKey)) {
            return false;
        }
        // 只要key相等，就在优先投放列表
        return StringUtils.equals(requestGroupKey, serverGroupKey);
    }

    /**
     * 是否无分组服务
     *
     * @param serverGroupKey
     * @return
     */
    private boolean isNoGroup(String serverGroupKey) {
        return StringUtils.isBlank(serverGroupKey);
    }

}
