package cn.com.duibaboot.ext.autoconfigure.perftest;

import cn.com.duiba.boot.perftest.PerfTestConstant;
import cn.com.duiba.boot.perftest.PerfTestUtils;
import cn.com.duiba.wolf.redis.RedisClient;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataRegister;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.etcd.client.EtcdKVClientDelegate;
import cn.com.duibaboot.ext.autoconfigure.javaagent.AttachJavaAgentListener;
import cn.com.duibaboot.ext.autoconfigure.perftest.core.PerfTestEnvCondition;
import cn.com.duibaboot.ext.autoconfigure.perftest.core.PerfTestFootMarker;
import cn.com.duibaboot.ext.autoconfigure.perftest.core.PerfTestProperties;
import cn.com.duibaboot.ext.autoconfigure.perftest.datasource.PerfTestDataSourcePostProcessor;
import cn.com.duibaboot.ext.autoconfigure.perftest.datasource.PerfTestRoutingDataSource;
import cn.com.duibaboot.ext.autoconfigure.perftest.filter.PerfTestFootFilter;
import cn.com.duibaboot.ext.autoconfigure.perftest.filter.ReactivePerfTestFootFilter;
import cn.com.duibaboot.ext.autoconfigure.perftest.hazelcast.PerfTestEurekaHazelcastGroupKey;
import cn.com.duibaboot.ext.autoconfigure.perftest.httpclient.PerfTestHttpAsyncClientBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.perftest.httpclient.PerfTestHttpClientBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.perftest.redis.jedis.PerfTestRedisCachePlugin;
import cn.com.duibaboot.ext.autoconfigure.perftest.redis.jedis.PerfTestSpringDataRedisPlugin;
import cn.com.duibaboot.ext.autoconfigure.perftest.redis.lettuce.LettuceConnectionFactoryBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.perftest.redis.lettuce.LettuceConnectionFactoryMethodInterceptor;
import com.hazelcast.core.HazelcastInstance;
import com.netflix.appinfo.EurekaInstanceConfig;
import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.nio.client.HttpAsyncClient;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnResource;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import redis.clients.jedis.Jedis;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.servlet.DispatcherType;
import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;

/**
 * 加载压测需要的相关Bean
 */
@Configuration
@EnableAspectJAutoProxy(proxyTargetClass = true)
@EnableConfigurationProperties(PerfTestProperties.class)
@ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
public class DuibaPerftestAutoConfiguration {

    @PostConstruct
    public void init() {
        if ("true".equals(System.getProperty(AttachJavaAgentListener.DUIBA_NOAGENT, "false"))) {
            throw new IllegalStateException("当关闭javaagent时禁止引入spring-boot-starter-perftest包, 如需开启压测支持，请开启javaagent，否则压测时可能会导致干扰生产数据。");
        }
    }

    /**
     * 压测足迹的注册，依赖etcd客户端
     */
    @Configuration
    @ConditionalOnBean(EtcdKVClientDelegate.class)
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestFootMarkerConfiguration {

        @Bean
        public PerfTestFootMarker perfTestFootMarker(EtcdKVClientDelegate etcdKVClientDelegate) {
            return new PerfTestFootMarker(etcdKVClientDelegate);
        }

    }

    /**
     * 注册元数据到Eureka中，表示当前cloud应用支持压测
     * 用于兼容老的压测ribbon判断逻辑，不加的话，如果客户端是老的ext，压测会直接报错，找不到下游
     *
     * @return
     */
    @Bean
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    public DiscoveryMetadataRegister perftestSupportDiscoveryMetadataRegister() {
        return appMetadata -> appMetadata.put(PerfTestConstant.IS_PERF_TEST_SUPPORTTED_KEY_OLD, "1");
    }

    /**
     * 把环境变量中的场景id写到准备注册到eureka的metadata中
     */
    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnClass(EurekaInstanceConfig.class)
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestDiscoveryMetadataRegisterConfiguration {

        @Bean
        public DiscoveryMetadataRegister perftestDiscoveryMetadataRegister() {
            return appMetadata -> {
                appMetadata.put(PerfTestConstant.PERF_TEST_SCENE_ID_EUREKA_KEY, PerfTestUtils.getSceneId());
                appMetadata.put(PerfTestConstant.PERF_TEST_SCENE_ID_EUREKA_KEY_OLD, PerfTestUtils.getSceneId());     // 兼容老的流程，防止未升级的应用调用错乱（防止生产流量调用到压测机器）
            };
        }

    }

    /**
     * 把环境变量中的场景id写到准备注册到eureka的metadata中
     */
    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    @ConditionalOnClass({ EurekaInstanceConfig.class, HazelcastInstance.class })
    static class PerfTestHazelcastEurekaInstanceConfigBeanPostProcessorConfiguration {

        @Bean
        public PerfTestEurekaHazelcastGroupKey perfTestEurekaHazelcastGroupKey() {
            String sceneId = PerfTestUtils.getSceneId();
            return new PerfTestEurekaHazelcastGroupKey(sceneId);
        }
    }

    /**
     * filter中加入压测足迹
     */
    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestFilterConfigurationServlet {

        @Bean
        public PerfTestFootFilter perfTestFootFilter() {
            return new PerfTestFootFilter();
        }

        @Bean
        public FilterRegistrationBean<PerfTestFootFilter> perfTestFootFilterConfigurer(PerfTestFootFilter perfTestFootFilter) {
            FilterRegistrationBean<PerfTestFootFilter> registrationBean = new FilterRegistrationBean<>();
            registrationBean.setFilter(perfTestFootFilter);
            List<String> urlPatterns = new ArrayList<>();
            urlPatterns.add("/*");  // 拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(Ordered.LOWEST_PRECEDENCE);
            return registrationBean;
        }

    }

    /**
     * filter中加入压测足迹
     */
    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.REACTIVE)
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestFilterConfigurationReactive {

        @Bean
        public ReactivePerfTestFootFilter reactivePerfTestFootFilter() {
            return new ReactivePerfTestFootFilter();
        }

    }

    /**
     * 把spring上下文中的所有数据源转换为自动路由的数据源，可以自动把压测流量导入影子库。
     */
    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnClass({ AbstractRoutingDataSource.class })
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestDataSourceConfiguration {

        @Resource
        private ApplicationContext applicationContext;

        @EventListener(EnvironmentChangeEvent.class)
        public void onEnvironmentChange(EnvironmentChangeEvent event) {
            Map<String, PerfTestRoutingDataSource> map = applicationContext.getBeansOfType(PerfTestRoutingDataSource.class);
            map.values().forEach(perfTestRoutingDataSource -> perfTestRoutingDataSource.onEnvironmentChange(event));
        }

        @Bean
        public static SpecifiedBeanPostProcessor<DataSource> perfTestDataSourcePostProcessor() {
            return new PerfTestDataSourcePostProcessor();
        }

    }

    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnClass({ RedisClient.class, Jedis.class, Aspect.class })
    @ConditionalOnBean({ RedisClient.class })
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestRedisCachePluginConfiguration {

        @Bean
        public PerfTestRedisCachePlugin newPerfTestRedisCachePlugin() {
            return new PerfTestRedisCachePlugin();
        }

    }

    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnClass({ JedisConnectionFactory.class, Jedis.class, Aspect.class })
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestSpringDataRedisPluginConfiguration {

        @Bean
        public PerfTestSpringDataRedisPlugin newPerfTestSpringDataRedisPlugin() {
            return new PerfTestSpringDataRedisPlugin();
        }

    }

    /**
     * 支持lettuce的压测
     */
    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnClass({ LettuceConnectionFactory.class, io.lettuce.core.RedisClient.class, Aspect.class })
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestSpringDataLettucePluginConfiguration {

        @Bean
        public static LettuceConnectionFactoryBeanPostProcessor newLettuceConnectionFactoryBeanPostProcessor() {
            return new LettuceConnectionFactoryBeanPostProcessor();
        }

        /**
         * 初始化这个bean只是为了调用该bean的onMainContextRefreshedEvent方法做些检查
         *
         * @return
         */
        @Bean
        public LettuceConnectionFactoryMethodInterceptor newLettuceConnectionFactoryMethodInterceptor() {
            return new LettuceConnectionFactoryMethodInterceptor();
        }

    }

    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnClass(HttpClient.class)
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestHttpClientBeanPostProcessorConfiguration {

        @Bean
        public static SpecifiedBeanPostProcessor<CloseableHttpClient> perfTestHttpClientBeanPostProcessor() {
            return new PerfTestHttpClientBeanPostProcessor();
        }
    }

    @Configuration
    @Conditional(PerfTestEnvCondition.class)
    @ConditionalOnClass(HttpAsyncClient.class)
    @ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
    static class PerfTestHttpAsyncClientBeanPostProcessorConfiguration {

        @Bean
        public static SpecifiedBeanPostProcessor<CloseableHttpAsyncClient> perfTestHttpAsyncClientBeanPostProcessor() {
            return new PerfTestHttpAsyncClientBeanPostProcessor();
        }
    }

}
