package cn.com.duibaboot.ext.autoconfigure.perftest.core;

import cn.com.duiba.boot.perftest.PerfTestUtils;
import cn.com.duibaboot.ext.autoconfigure.etcd.client.EtcdKVClientDelegate;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

public class PerfTestFootMarker {

    private static final Logger LOGGER = LoggerFactory.getLogger(PerfTestFootMarker.class);

    private static final String PERF_TEST_PREFIX         = "/perftest/{sceneId}/";
    private static final String PERF_TEST_APPS_FOOT_KEY  = "/perftest/{sceneId}/apps/{appName}";
    private static final String PERF_TEST_DB_FOOT_KEY    = "/perftest/{sceneId}/dbUrl/{url}";
    private static final String PERF_TEST_REDIS_FOOT_KEY = "/perftest/{sceneId}/redisUrl/{url}";
    private static final String SCENE_ID_PLACEHOLDER     = "{sceneId}";
    private static final String APP_NAME_PLACEHOLDER     = "{appName}";
    private static final String URL_PLACEHOLDER          = "{url}";
    private static final String PERF_TEST_FOOT_VALUE     = "default";

    @Value("${spring.application.name}")
    private String currentAppName;

    private final EtcdKVClientDelegate etcdKVClientDelegate;

    private final LoadingCache<String, AtomicBoolean> puttingFlagCache = Caffeine.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).initialCapacity(5).maximumSize(100).build(key -> new AtomicBoolean(false));

    private final Cache<String, String> perfTestFootMarkerCache = Caffeine.newBuilder().expireAfterWrite(30, TimeUnit.SECONDS).initialCapacity(5).maximumSize(100).build();

    @PostConstruct
    public void init() {
        PerfTestFootMarkerHolder._setPerfTestFootMarker(this);
    }

    private AtomicBoolean getPuttingFlag(String key) {
        return puttingFlagCache.get(key);
    }

    private String getCache(String key) {
        String value = null;
        try {
            value = perfTestFootMarkerCache.get(key, k -> etcdKVClientDelegate.get(key));
        } catch (Exception e) {
            LOGGER.error("获取压测足迹异常，key={}", key, e);
        }
        return value;
    }

    private void putCache(String key, String value) {
        try {
            etcdKVClientDelegate.put(key, value);
            perfTestFootMarkerCache.put(key, value);
        } catch (Exception e) {
            LOGGER.error("存储压测足迹异常，key={}", key, e);
        }
    }

    public PerfTestFootMarker(EtcdKVClientDelegate etcdKVClientDelegate) {
        this.etcdKVClientDelegate = etcdKVClientDelegate;
    }

    private void mark(String halfKey) {
        String sceneId = PerfTestUtils.getSceneId();
        if (sceneId == null) {
            return;
        }
        String key = halfKey.replace(SCENE_ID_PLACEHOLDER, sceneId);
        String value = perfTestFootMarkerCache.getIfPresent(key);
        if (StringUtils.isNotBlank(value)) {
            return;
        }
        AtomicBoolean putting = this.getPuttingFlag(key);
        if (putting == null) {
            LOGGER.warn("getPuttingFlag error, key={}", key);
            return;
        }
        if (putting.compareAndSet(false, true)) {
            try {
                value = this.getCache(key);
                if (StringUtils.isBlank(value)) {
                    value = PERF_TEST_FOOT_VALUE;
                    this.putCache(key, value);
                }
            } finally {
                putting.set(false);
            }
        }
    }

    /**
     * 标记压测流量经过的应用名称
     * 注意：除非特殊情况，否则框架使用者不应该调用此接口
     */
    public void markApp() {
        // 当前不是压测环境
        if (!PerfTestUtils.isPerfTestEnv()) {
            return;
        }
        if (StringUtils.isBlank(currentAppName)) {
            return;
        }
        this.mark(PERF_TEST_APPS_FOOT_KEY.replace(APP_NAME_PLACEHOLDER, currentAppName));
    }

    /**
     * 标记压测流量经过的数据库url
     * 注意：除非特殊情况，否则框架使用者不应该调用此接口
     *
     * @param url
     */
    public void markDb(String url) {
        // 当前不是压测环境
        if (!PerfTestUtils.isPerfTestEnv()) {
            return;
        }
        if (StringUtils.isBlank(url)) {
            return;
        }
        this.mark(PERF_TEST_DB_FOOT_KEY.replace(URL_PLACEHOLDER, url));
    }

    /**
     * 标记压测流量经过的redis url
     * 注意：除非特殊情况，否则框架使用者不应该调用此接口
     *
     * @param url
     */
    public void markRedis(String url) {
        // 当前不是压测环境
        if (!PerfTestUtils.isPerfTestEnv()) {
            return;
        }
        if (StringUtils.isBlank(url)) {
            return;
        }
        this.mark(PERF_TEST_REDIS_FOOT_KEY.replace(URL_PLACEHOLDER, url));
    }

    /**
     * 根据场景id，删除压测足迹
     * 注意：用于hulk-web调用，其他框架使用者请不要调用
     *
     * @param sceneId
     */
    public void removeMark(String sceneId) {
        if (StringUtils.isBlank(sceneId)) {
            return;
        }
        try {
            String prefix = PERF_TEST_PREFIX.replace(SCENE_ID_PLACEHOLDER, sceneId);
            etcdKVClientDelegate.deleteWithPrefix(prefix);
        } catch (Exception e) {
            LOGGER.error("清除etcd场景压测足迹异常，sceneId={}", sceneId, e);
        }
    }

    /**
     * 根据场景id获取压测流量足迹的标记
     * 注意：用于hulk-web调用，其他框架使用者请不要调用
     *
     * @param sceneId
     * @return
     */
    public Map<String, String> getMarks(String sceneId) {
        if (StringUtils.isBlank(sceneId)) {
            return Collections.emptyMap();
        }
        try {
            String prefix = PERF_TEST_PREFIX.replace(SCENE_ID_PLACEHOLDER, sceneId);
            return etcdKVClientDelegate.getWithPrefix(prefix);
        } catch (Exception e) {
            LOGGER.error("获取etcd场景压测足迹异常，sceneId={}", sceneId, e);
        }
        return Collections.emptyMap();
    }
}
