package cn.com.duibaboot.ext.autoconfigure.perftest.cloud.netflix.ribbon;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duiba.boot.perftest.PerfTestConstant;
import cn.com.duibaboot.ext.autoconfigure.perftest.core.UnsupportedAppsHolder;
import cn.com.duiba.boot.perftest.PerfTestUtils;
import com.netflix.loadbalancer.Server;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.annotation.Order;

import java.util.ArrayList;
import java.util.List;

/**
 * 压测实例的ribbon过滤器
 * 这个Bean就算不是压测机器，也要注册
 */
@Order(0)
public class PerfTestRibbonServerListFilter implements RibbonServerListFilter {

    @Override
    public List<Server> filter(List<Server> serverList, Object key) {
        if (serverList.isEmpty()) {
            return serverList;
        }
        for (Server s : serverList) {
            if (!(s instanceof DiscoveryEnabledServer)) {
                return serverList;
            }
        }

        // 当前机器环境变量中的场景id
        String currentSceneId = PerfTestUtils.getSceneId();
        String targetAppName = null;

        List<Server> testList = new ArrayList<>(serverList.size());
        List<Server> prodList = new ArrayList<>(serverList.size());
        for (Server server : serverList) {
            String targetSceneId = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(PerfTestConstant.PERF_TEST_SCENE_ID_EUREKA_KEY);
            if (StringUtils.isBlank(targetAppName)) {
                targetAppName = server.getMetaInfo().getAppName().toLowerCase();
            }

            if (sceneIdEquals(currentSceneId, targetSceneId)) { // 本机器和目标服务器有相同的场景id
                testList.add(server);
            }

            if (StringUtils.isBlank(targetSceneId)) {   // 生产的机器
                prodList.add(server);
            }
        }

        // 独立容器list不为空
        if (CollectionUtils.isNotEmpty(testList)) {
            return testList;
        }

        // 当前机器是压测机器，但是找不到相同场景id的压测机器，并且目标应用不在不支持压测的应用列表中，那么直接把所有机器都过滤掉，让他外层找不到服务报错
        if (StringUtils.isNotBlank(currentSceneId) && !UnsupportedAppsHolder.isUnsupportedApp(targetAppName)) {
            throw new IllegalStateException("当前机器是压测机器，但是找不到相同场景id的压测机器。并且目标应用：" + targetAppName + " 不在【unsupportApps】中，流程无法继续往下调用");
        }

        // 其他情况走生产（一般生产机器不会没有，如果生产机器没有，那么随便调用一台）
        return CollectionUtils.isNotEmpty(prodList) ? prodList : serverList;
    }

    /**
     * 判断当前服务器环境变量的场景id和服务器注册在eureka的场景id是否相同
     * 任何一个为null，都认为不相同
     *
     * @param currentSceneId
     * @param targetSceneId
     * @return
     */
    private boolean sceneIdEquals(String currentSceneId, String targetSceneId) {
        if (currentSceneId == null || targetSceneId == null) {
            return false;
        }
        return currentSceneId.equals(targetSceneId);
    }
}
