package cn.com.duibaboot.ext.autoconfigure.perftest.log;

import ch.qos.logback.classic.AsyncAppender;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.ConsoleAppender;
import ch.qos.logback.core.FileAppender;
import cn.com.duiba.boot.perftest.PerfTestUtils;
import cn.com.duibaboot.ext.autoconfigure.perftest.core.PerfTestEnvCondition;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnResource;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;

import javax.annotation.Resource;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Created by guoyanfei .
 * 2022/2/25 .
 */
@Configuration
@ConditionalOnClass({ LoggerContext.class })
@Conditional(PerfTestEnvCondition.class)
@ConditionalOnResource(resources = "classpath:autoconfig/perftest.properties")
public class PerfTestLogAutoConfiguration {

    private static final Logger LOGGER = LoggerFactory.getLogger(PerfTestLogAutoConfiguration.class);

    @Bean
    public PerfTestLogNameFilter perfTestLogNameFilter() {
        return new PerfTestLogNameFilter();
    }

    @Bean
    public PerfTestLogDenyFilter perfTestLogDenyFilter() {
        return new PerfTestLogDenyFilter();
    }

    @Bean
    public LogApplicationListener logApplicationListener() {
        return new LogApplicationListener();
    }

    private static class LogApplicationListener implements ApplicationListener<ContextRefreshedEvent>, Ordered {

        private static volatile boolean flag = true;

        //用Set保证多个Logger注入同一个Appender时，FileAppender不会重复注入计数器
        private final Set<FileAppender>    fileAppenders    = Sets.newHashSet();
        private final Set<ConsoleAppender> consoleAppenders = Sets.newHashSet();
        private final Set<AsyncAppender>   asyncAppenders   = Sets.newHashSet();

        @Resource
        private PerfTestLogNameFilter perfTestLogNameFilter;

        @Resource
        private PerfTestLogDenyFilter perfTestLogDenyFilter;

        // 要晚于LoggerAutoConfiguration.AsyncLoggerConfiguarListener执行
        @Override
        public int getOrder() {
            return Ordered.LOWEST_PRECEDENCE - 20;
        }

        @Override
        public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
            if (!PerfTestUtils.isPerfTestEnv()) {
                return;
            }
            if (!flag) {
                return;
            }
            try {
                LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();

                List<ch.qos.logback.classic.Logger> loggers = loggerContext.getLoggerList();
                for (ch.qos.logback.classic.Logger logger : loggers) {
                    collectAppenders(logger);
                }

                for (FileAppender fileAppender : fileAppenders) {
                    if (fileAppender.getFile().contains("application")) {
                        fileAppender.addFilter(perfTestLogNameFilter); // 压测的日志，线程名字带上压测字样
                    } else {
                        fileAppender.addFilter(perfTestLogDenyFilter); // 压测的时候，除了application日志之外，其他所有日志都不打印
                    }
                }

                for (ConsoleAppender consoleAppender : consoleAppenders) {
                    consoleAppender.addFilter(perfTestLogNameFilter); // 压测的日志，线程名字带上压测字样
                }

                for (AsyncAppender asyncAppender : asyncAppenders) {
                    Iterator<Appender<ILoggingEvent>> iterator = asyncAppender.iteratorForAppenders();
                    while (iterator.hasNext()) {
                        Appender<ILoggingEvent> childAsyncAppender = iterator.next();
                        if (childAsyncAppender instanceof FileAppender) {
                            FileAppender fileAppender  = (FileAppender) childAsyncAppender;
                            if (fileAppender.getFile().contains("application")) {
                                asyncAppender.addFilter(perfTestLogNameFilter); // 压测的日志，线程名字带上压测字样
                            } else {
                                asyncAppender.addFilter(perfTestLogDenyFilter); // 压测的时候，除了application日志之外，其他所有日志都不打印
                            }
                        }
                    }
                }
            } catch (Exception e) {
                LOGGER.error(e.getMessage(), e);
            }
            flag = false;
        }

        /**
         * 把log的Appenders收集起来
         *
         * @param log
         */
        private void collectAppenders(ch.qos.logback.classic.Logger log) {
            Iterator<Appender<ILoggingEvent>> iter = log.iteratorForAppenders();

            while (iter.hasNext()) {
                Appender<ILoggingEvent> appender = iter.next();
                if (appender instanceof FileAppender) {
                    fileAppenders.add((FileAppender) appender);
                }
                if (appender instanceof ConsoleAppender) {
                    consoleAppenders.add((ConsoleAppender) appender);
                }
                if (appender instanceof AsyncAppender) {
                    Iterator<Appender<ILoggingEvent>> iterator = ((AsyncAppender) appender).iteratorForAppenders();
                    while (iterator.hasNext()) {
                        Appender<ILoggingEvent> childAsyncAppender = iterator.next();
                        if (childAsyncAppender instanceof FileAppender) {
                            fileAppenders.add((FileAppender) childAsyncAppender);
                        }
                    }
                    asyncAppenders.add((AsyncAppender) appender);
                }
            }
        }

    }

}
