package cn.com.duibaboot.ext.autoconfigure.perftest.redis.lettuce;

import cn.com.duiba.boot.perftest.PerfTestUtils;
import io.lettuce.core.resource.DefaultClientResources;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.DestructionAwareBeanPostProcessor;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettucePoolingClientConfiguration;

import java.util.HashMap;
import java.util.Map;

/**
 * 给压测和正常流量分别设置两个隔离的LettuceConnectionFactory。（这要求所有LettuceConnectionFactory都要定义在spring bean上下文中，所以需要配合LettuceConnectionFactoryInstrumentation类来判断bean是不是真的定义在了spring上下文中，否则压测流量会打到线上，导致问题）
 */
public class LettuceConnectionFactoryBeanPostProcessor implements DestructionAwareBeanPostProcessor {

    private static final Map<LettuceConnectionFactory, LettuceConnectionFactory> LETTUCE_CONNECTION_FACTORY_BEAN_MAP = new HashMap<>();

    @Override
    public void postProcessBeforeDestruction(Object bean, String beanName) throws BeansException {
        if (bean.getClass() == LettuceConnectionFactory.class) {
            LettuceConnectionFactory perfTestBean = LETTUCE_CONNECTION_FACTORY_BEAN_MAP.get(bean);
            if (perfTestBean == null) {
                return;
            }
            LETTUCE_CONNECTION_FACTORY_BEAN_MAP.remove(bean);

            // clientResources 如果不关闭，在DefaultClientResources.create()执行多次后，会报错 LEAK: HashedWheelTimer.release() was not called before it's garbage-collected.
            try {
                perfTestBean.getClientResources().shutdown().get();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            // 销毁方法，在确保执行 normalBean.destroy() 的时候，也需要调用 perfTestBean.destroy();
            perfTestBean.destroy();
        }
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        if (bean.getClass() == LettuceConnectionFactory.class) {
            // 给正常流量使用的LettuceConnectionFactory
            LettuceConnectionFactory normalBean = (LettuceConnectionFactory) bean;
            // 给压测流量使用的LettuceConnectionFactory perfTestBean
            LettuceConnectionFactory perfTestBean = getPerfTestBean(normalBean);
            perfTestBean.setDatabase(normalBean.getDatabase() + 1);

            LETTUCE_CONNECTION_FACTORY_BEAN_MAP.put(normalBean, perfTestBean);

            ProxyFactory factory = new ProxyFactory();
            factory.setTarget(normalBean);
            factory.addAdvice(new LettuceConnectionFactoryMethodInterceptor(normalBean, perfTestBean));
            return factory.getProxy();
        }
        return bean;
    }

    private LettuceConnectionFactory getPerfTestBean(LettuceConnectionFactory normalBean) {
        RedisStandaloneConfiguration config = new RedisStandaloneConfiguration();
        config.setHostName(normalBean.getHostName());
        config.setPort(normalBean.getPort());
        config.setPassword(RedisPassword.of(normalBean.getPassword()));
        config.setDatabase(normalBean.getDatabase());
        DefaultClientResources clientResources = DefaultClientResources.create();
        GenericObjectPoolConfig poolConfig = new GenericObjectPoolConfig();
        poolConfig.setMaxTotal(8);
        poolConfig.setMinIdle(1);
        poolConfig.setMaxIdle(8);
        poolConfig.setMaxWaitMillis(100);
        poolConfig.setTestWhileIdle(true);
        poolConfig.setTimeBetweenEvictionRunsMillis(180000);//每隔180S testWhileIdle一次
        LettucePoolingClientConfiguration lettucePoolingClientConfiguration = LettucePoolingClientConfiguration.builder().poolConfig(poolConfig).clientResources(clientResources).commandTimeout(normalBean.getClientConfiguration().getCommandTimeout()).shutdownTimeout(normalBean.getClientConfiguration().getShutdownTimeout()).build();

        CustomLettuceConnectionFactory perfTestBean = new CustomLettuceConnectionFactory(config, lettucePoolingClientConfiguration);
        perfTestBean.setConvertPipelineAndTxResults(normalBean.getConvertPipelineAndTxResults());
        perfTestBean.setShareNativeConnection(normalBean.getShareNativeConnection());
        perfTestBean.setValidateConnection(normalBean.getValidateConnection());

        return perfTestBean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    private static class LettuceConnectionFactoryMethodInterceptor implements MethodInterceptor {

        private final LettuceConnectionFactory normalBean;
        private final LettuceConnectionFactory perfTestBean;

        LettuceConnectionFactoryMethodInterceptor(LettuceConnectionFactory normalBean, LettuceConnectionFactory perfTestBean) {
            this.normalBean = normalBean;
            this.perfTestBean = perfTestBean;
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            String methodName = invocation.getMethod().getName();

            if (methodName.equals("afterPropertiesSet")) {
                //如果是初始化方法，两个LettuceConnectionFactory实例都要确保调用到
                perfTestBean.afterPropertiesSet();
                return invocation.proceed();//invocation.proceed()实际上会调用normalBean的afterPropertiesSet/destroy方法
            } else {
                if (!PerfTestUtils.isPerfTestEnv()) {
                    //正常流量直接转发给normalBean即可
                    return invocation.proceed();
                } else {
                    //压测流量把方法调用转发给perfTestBean，从而支持压测时获取不同的connection
                    return invocation.getMethod().invoke(perfTestBean, invocation.getArguments());
                }
            }
        }

    }

}
