package cn.com.duibaboot.ext.autoconfigure.sleuth;

import brave.Span;
import brave.baggage.BaggageField;
import brave.baggage.BaggagePropagation;
import brave.baggage.BaggagePropagationConfig;
import brave.propagation.B3Propagation;
import brave.propagation.CurrentTraceContext;
import brave.propagation.Propagation;
import cn.com.duiba.wolf.redis.RedisClient;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duiba.boot.grouping.ServiceGroupUtils;
import cn.com.duibaboot.ext.autoconfigure.sleuth.ttl.ThreadLocalCurrentTraceContext;
import com.alibaba.ttl.TransmittableThreadLocal;
import io.lettuce.core.RedisFuture;
import io.searchbox.client.JestClient;
import org.apache.commons.collections.CollectionUtils;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.session.SqlSessionFactory;
import org.aspectj.lang.annotation.Aspect;
import org.elasticsearch.client.transport.TransportClient;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import redis.clients.jedis.Jedis;

/**
 * Created by wenqi.huang on 2016/11/7.
 */
@Configuration
@EnableAspectJAutoProxy(proxyTargetClass = true)
public class SleuthAutoConfiguration {

    @Configuration
    @ConditionalOnClass({SqlSessionTemplate.class, SqlSessionFactoryBean.class, SqlSessionFactory.class, Span.class})
    public static class SleuthMyBatisPostProcessorConfiguration{

        @Bean
//        @Scope("prototype") //如果aspect类中有状态，则需要 设置成prototype
        public SleuthMybatisPlugin sleuthMybatisPlugin(){
            return new SleuthMybatisPlugin();
        }

        /**
         * 声明后置处理器，spring全部bean初始化完成后调用，给所有SqlSessionBean注入SleuthMybatisPlugin plugin，监控sql的执行
         * @return
         */
        @Bean
        public static SpecifiedBeanPostProcessor sleuthMyBatisPostProcessorConfigurer(){
            return new SleuthMybatisBeanPostProcessor();
        }
    }

    private static class SleuthMybatisBeanPostProcessor implements SpecifiedBeanPostProcessor<Object>, BeanFactoryAware {
        private BeanFactory beanFactory;
        private volatile SleuthMybatisPlugin sleuthMybatisPlugin;

        private SleuthMybatisPlugin getSleuthMybatisPlugin() {
            if(sleuthMybatisPlugin == null) {
                sleuthMybatisPlugin = beanFactory.getBean(SleuthMybatisPlugin.class);
            }
            return sleuthMybatisPlugin;
        }

        @Override
        public int getOrder() {
            return -1;
        }

        @Override
        public Class<Object> getBeanType() {
            return Object.class;
        }

        @Override
        public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
            return bean;
        }

        @Override
        public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
            SqlSessionFactory s = null;
            if(bean instanceof SqlSessionFactory){
                s = (SqlSessionFactory)bean;
            }
            if(bean instanceof SqlSessionTemplate){
                s = ((SqlSessionTemplate)bean).getSqlSessionFactory();
            }
            if(s == null){
                return bean;
            }

            addInterceptor(s);

            return bean;
        }

        private void addInterceptor(SqlSessionFactory s){
            boolean hasCatPlugin = false;
            if(!CollectionUtils.isEmpty(s.getConfiguration().getInterceptors())) {
                for (Interceptor plugin : s.getConfiguration().getInterceptors()) {
                    if (plugin instanceof SleuthMybatisPlugin) {
                        hasCatPlugin = true;
                        break;
                    }
                }
            }

            if (!hasCatPlugin && getSleuthMybatisPlugin() != null) {
                s.getConfiguration().addInterceptor(getSleuthMybatisPlugin());
            }
        }

        @Override
        public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
            this.beanFactory = beanFactory;
        }
    }

    @Configuration
    @ConditionalOnClass({RedisClient.class, Jedis.class, Span.class, Aspect.class})
    public static class SleuthRedisPluginConfiguration{

    	@Bean
    	public SleuthRedisPlugin sleuthRedisPlugin(){
    		return new SleuthRedisPlugin();
    	}

    }

    /**
     * 加入aop，监控spring-data-redis执行耗时
     */
    @Configuration
    @ConditionalOnClass({JedisConnectionFactory.class, Jedis.class, Span.class, Aspect.class})
    public static class SleuthSpringDataRedisAspectConfiguration{

    	@Bean
    	public SleuthSpringDataRedisPlugin sleuthSpringDataRedisAspectPlugin(){
    		return new SleuthSpringDataRedisPlugin();
    	}

    }

    /**
     * 加入aop，监控spring-data-redis执行耗时
     */
    @Configuration
    @ConditionalOnClass({JedisConnectionFactory.class, RedisFuture.class, Span.class, Aspect.class})
    public static class SleuthSpringDataLettuceAspectConfiguration{

    	@Bean
    	public SleuthSpringDataLettucePlugin sleuthSpringDataLettuceAspectPlugin(){
    		return new SleuthSpringDataLettucePlugin();
    	}

    }

//    @Configuration
//    @ConditionalOnClass({XMemcacheClient.class, MemcachedClient.class, Span.class, Aspect.class})
//    public static class SleuthMemcacheConfiguration{
//
//    	@Bean
//    	public SleuthMemcachePlugin sleuthMemcachePlugin(){
//    		return new SleuthMemcachePlugin();
//    	}
//
//    }


    @Configuration
    @ConditionalOnClass({Span.class, Aspect.class,MongoOperations.class})
    public static class SleuthMongodbConfiguration{

    	@Bean
    	public SleuthMongodbPlugin sleuthMongodbPlugin(){
    		return new SleuthMongodbPlugin();
    	}

    }

    @Configuration
    @ConditionalOnClass({TransportClient.class, ElasticsearchOperations.class, Span.class, Aspect.class})
    public static class SleuthElasticsearchConfiguration{

    	@Bean
    	public SleuthElasticSearchPlugin sleuthElasticSearchPlugin(){
    		return new SleuthElasticSearchPlugin();
    	}

    }

    @Configuration
    @ConditionalOnClass({JestClient.class, Span.class, Aspect.class})
    public static class SleuthJestElasticsearchConfiguration{
    	@Bean
        public SleuthJestElasticSearchPlugin sleuthJestElasticSearchPlugin(){
            return new SleuthJestElasticSearchPlugin();
    	}
    }

    @Configuration
    @ConditionalOnClass({Connection.class, HTable.class, Span.class, Aspect.class})
    public static class SleuthHbaseConfiguration{

        @Bean
        public SleuthHbasePlugin sleuthHbasePlugin(){
            return new SleuthHbasePlugin();
        }

    }

    /**
     * sleuth线程传递修改为ttl（ttl使用javaagent启用）
     */
    @Configuration
    @ConditionalOnClass({TransmittableThreadLocal.class})
    public static class SleuthTtlConfiguration{

        @Bean
        public CurrentTraceContext.Builder sleuthCurrentTraceContextBuilder() {
            return ThreadLocalCurrentTraceContext.newBuilder();
        }

    }

    /**
     * 支持sleuth前缀透传
     *
     * @see org.springframework.cloud.sleuth.autoconfig.TraceBaggageConfiguration#baggagePropagationFactoryBuilder() 可以参考sleuth的原生注入
     */
    @Configuration
    @ConditionalOnClass({BaggagePropagation.FactoryBuilder.class})
    public static class SleuthPropagationConfiguration {

        static final Propagation.Factory B3_FACTORY = B3Propagation.newFactoryBuilder()
                .injectFormat(B3Propagation.Format.SINGLE_NO_PARENT).build();

        @Bean
        public BaggagePropagation.FactoryBuilder baggagePropagationFactoryBuilder() {
            BaggagePropagation.FactoryBuilder factoryBuilder = BaggagePropagation.newFactoryBuilder(B3_FACTORY);
            // 默认加入 _duibaServiceGroupKey 透传
            factoryBuilder.add(BaggagePropagationConfig.SingleBaggageField.remote(BaggageField.create(ServiceGroupUtils.DUIBA_SERVICE_GROUP_KEY)));
            return factoryBuilder;
        }
    }

}
