/**
 * Copyright (c) 2022, duiba.com.cn All Rights Reserved.
 */
package cn.com.duibaboot.ext.autoconfigure.monitor.rpc.scan;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 描述: rpc 引用扫描
 *
 * @author guopengfei@duiba.com.cn
 * @version v1.0
 * @date 2022/2/25 16:25
 */
public class RpcConsumerScan {

    private static final Logger LOGGER = LoggerFactory.getLogger(RpcConsumerScan.class);

    private static final List<Class> INJECT_CLASS_LIST = new ArrayList<>();

    static {
        INJECT_CLASS_LIST.add(Autowired.class);
        INJECT_CLASS_LIST.add(Resource.class);
    }

    private static final Map<String, Set<String>> RPC_CONSUMER_CONFIG = new ConcurrentHashMap<>();

    /**
     * 扫描引用了哪些rpc接口
     *
     * @param beanFactory
     * @param applicationName
     */
    public static void scanAndStore(ConfigurableListableBeanFactory beanFactory,
                                    String applicationName) {
        List<String> scanPackageList = AutoConfigurationPackages.get(beanFactory);

        for (String bd : beanFactory.getBeanDefinitionNames()) {
            BeanDefinition beanDefinition = beanFactory.getBeanDefinition(bd);
            String beanClassName = beanDefinition.getBeanClassName();
            if (beanClassName == null) {
                continue;
            }
            // 判断是否是 org.springframework.boot.autoconfigure.SpringBootApplication 的扫描包
            // 如果不是项目自定义的包，不再扫描()。
            // 由于有些jar，为了增加RPC接口缓存，直接在jar包内注入了容器，所以使用SpringBootApplication上的包集合作为白名单
            boolean isWhitePackages = scanPackageList.stream()
                    .anyMatch(packageName -> beanClassName.startsWith(packageName));
            if (!isWhitePackages) {
                continue;
            }
            Class<?> clazz = null;
            try {
                clazz = Class.forName(beanClassName);
            } catch (ClassNotFoundException e) {
                continue;
            }

            analyseAndRegister(applicationName, clazz);
        }
    }

    private static void analyseAndRegister(String applicationName, Class<?> clazz) {
        List<Field> fields = getFieldList(clazz);
        for (Field field : fields) {
            if (isInjectAnnotationPresent(field, INJECT_CLASS_LIST)) {
                // 由于本次方案，启动dubbo的接口，都默认启动的feign，所以所有的判断都基于 AdvancedFeignClient
                if (field.getType().isAnnotationPresent(AdvancedFeignClient.class)) {
                    String appName = injectValueFromPackages(field.getType().getPackage().getName());
                    registerRpcMonitor(appName, field.getType().getName(), applicationName);
                }
            }
        }
    }

    private static String injectValueFromPackages(String packageName){
        if(packageName == null){
            return "";
        }
        try {
            //如果还没有加载过某个包下的类，则Package肯定找不到，所以这里先初始化一下
            Class.forName(packageName + ".package-info");
        } catch (ClassNotFoundException e) {
            //Ignore
        }
        Package pkg = Package.getPackage(packageName);
        if(pkg == null){
            //递归地到上级package寻找
            return injectValueFromPackages(parentPackage(packageName));
        }
        AdvancedFeignClient[] properties = pkg.getAnnotationsByType(AdvancedFeignClient.class);
        if(!ArrayUtils.isEmpty(properties)){
            return properties[0].value();
        }
        //递归地到上级package寻找
        return injectValueFromPackages(parentPackage(packageName));
    }

    private static String parentPackage(String packageName){
        int idx = packageName.lastIndexOf('.');
        if(idx == -1){
            return null;
        }
        return packageName.substring(0, idx);
    }

    private static List<Field> getFieldList(Class<?> clazz) {
        if (null == clazz || Objects.equals(Object.class, clazz)) {
            return Collections.emptyList();
        }
        Field[] fields = clazz.getDeclaredFields();
        List<Field> fieldList = Lists.newArrayList(fields);
        Class<?> superClass = clazz.getSuperclass();
        if (superClass.equals(Object.class)) {
            return fieldList;
        }
        fieldList.addAll(getFieldList(superClass));
        return fieldList;
    }

    private static boolean isInjectAnnotationPresent(Field field, List<Class> injectClassList) {
        if (field == null) {
            return false;
        }
        for (Class clazz : injectClassList) {
            if (field.isAnnotationPresent(clazz)) {
                // 只要被任意的注解注入，就算引入了该类
                return true;
            }
        }
        return false;
    }

    /**
     * 注册rpc的监控数据
     *
     * @param application
     * @param interfaceName
     */
    public synchronized static void registerRpcMonitor(String application, String interfaceName, String applicationName) {
        if (StringUtils.isBlank(application)) {
            return;
        }
        if (application.equals(applicationName)) {
            return;
        }
        RPC_CONSUMER_CONFIG.compute(application, (k, v) -> {
            if (v == null) {
                Set<String> interfaceNameSet = new HashSet<>();
                interfaceNameSet.add(interfaceName);
                return interfaceNameSet;
            }
            v.add(interfaceName);
            return v;
        });
    }

    public static Map<String, Set<String>> getRpcMonitorInfo() {
        return RPC_CONSUMER_CONFIG;
    }

    /**
     * 注册注入类
     *
     * @param clazzList
     */
    public static void registerInjectClass(List<Class> clazzList) {
        if (CollectionUtils.isEmpty(clazzList)) {
            return;
        }
        INJECT_CLASS_LIST.addAll(clazzList);
    }

}
