package cn.com.duibaboot.ext.autoconfigure.perftest.redis.lettuce;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceConstructorInterceptor;
import org.springframework.context.event.EventListener;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;

import java.lang.ref.WeakReference;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * 检测LettuceConnectionFactory是不是在spring bean中定义的？如果不是则禁止启动（只有在bean中定义才能正确地启用对lettuce的压测功能）
 */
public class LettuceConnectionFactoryMethodInterceptor implements InstanceConstructorInterceptor {

    private static final Set<WeakReference<LettuceConnectionFactory>> lettuceConnectionFactorys = new HashSet<>();

    /**
     * checked为true时，表示【已经在应用刚启动完成的时候做过对lettuceConnectionFactorys数量的检测】
     */
    private static volatile boolean checked = false;

    @Override
    public synchronized void onConstruct(Object objInst, Object[] allArguments) {
        if (objInst instanceof CustomLettuceConnectionFactory) {//忽略CustomLettuceConnectionFactory实例
            return;
        }
        if (checked) {  // 已经在应用刚启动完成的时候做过对lettuceConnectionFactorys数量的检测。此时不用再检测（通常在调用refresh端点之后，也会进入此处逻辑，此时不需要再检测）
            return;
        }
        for (WeakReference<LettuceConnectionFactory> r : lettuceConnectionFactorys) {
            if (r.get() == objInst) {
                return;
            }
        }

        lettuceConnectionFactorys.add(new WeakReference<>((LettuceConnectionFactory) objInst));

        if (lettuceConnectionFactorys.size() > 20) {
            throw new IllegalStateException("你的程序初始化了太多个LettuceConnectionFactory实例！");
        }
    }

    @EventListener(MainContextRefreshedEvent.class)
    public void onMainContextRefreshedEvent(MainContextRefreshedEvent event){
        checked = true;
        Map<String, LettuceConnectionFactory> map = event.getApplicationContext().getBeansOfType(LettuceConnectionFactory.class);
        //判断beanFactory所有实例个数和构造的实例个数是不是相同！
        if(map.size() != lettuceConnectionFactorys.size()) {
            throw new IllegalStateException("请把LettuceConnectionFactory定义成spring bean，否则无法启用Lettuce的压测功能!");
        }
    }
}
