package cn.com.duibaboot.ext.autoconfigure.accesslog;

import cn.com.duiba.boot.perftest.ReactivePerfTestUtils;
import cn.com.duibaboot.ext.autoconfigure.core.utils.ReactiveHttpRequestUtils;
import org.springframework.http.HttpCookie;
import org.springframework.http.ResponseCookie;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.util.MultiValueMap;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilter;
import org.springframework.web.server.WebFilterChain;
import reactor.core.publisher.Mono;

import java.util.*;

/**
 * for reactive/webflux
 * Created by hwq
 */
public class AccessLogReactiveFilter extends AbstractAccessLogFilter<ServerWebExchange, ServerHttpResponse> implements WebFilter {

    public static final String OW_HOST=AbstractAccessLogFilter.OW_HOST;

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, WebFilterChain chain) {
        exchange.getRequest();
        long start = System.currentTimeMillis();
        doBefore(exchange, exchange.getResponse());

        return chain.filter(exchange).then(Mono.fromRunnable(() -> {
            doAfter(exchange, exchange.getResponse(), System.currentTimeMillis() - start);
        }));
    }

    @Override
    protected String getHost(ServerWebExchange req) {
        Map<String, String> map = req.getAttribute(OVERWRITE_ATTRIBUTE_KEY);
        if(map!=null){
            String host=map.get(OW_HOST);
            if(host!=null){
                return host;
            }
        }

        return getHeader(req, "host");
    }

    @Override
    protected String getRequestURI(ServerWebExchange req) {
        return req.getRequest().getURI().getPath();
    }

    @Override
    protected String getMethod(ServerWebExchange req) {
        return req.getRequest().getMethodValue();
    }

    @Override
    protected String getQueryString(ServerWebExchange req) {
        return req.getRequest().getURI().getQuery();
    }

    @Override
    protected String getHeader(ServerWebExchange req, String key) {
        return req.getRequest().getHeaders().getFirst(key);
    }

    @Override
    protected String getIpAddr(ServerWebExchange req) {
        return ReactiveHttpRequestUtils.getIpAddr(req.getRequest());
    }

    @Override
    protected void addAttribute(ServerWebExchange req, String key, Object value) {
        req.getAttributes().put(key, value);
    }

    @Override
    protected Object getAttribute(ServerWebExchange req, String key) {
        return req.getAttribute(key);
    }

    @Override
    protected String getParameter(ServerWebExchange req, String key) {
        return req.getRequest().getQueryParams().getFirst(key);
    }

    @Override
    protected List<String> getParameterNames(ServerWebExchange req) {
        MultiValueMap<String, String> map = req.getRequest().getQueryParams();
        return new ArrayList<>(map.keySet());
    }

    @Override
    protected int getStatus(ServerHttpResponse resp) {
        return resp.getStatusCode() == null ? 200 : resp.getStatusCode().value();
    }

    @Override
    protected void addCookie(ServerHttpResponse resp, String key, String value, String domain, String path) {
        ResponseCookie.ResponseCookieBuilder cookieBuilder = ResponseCookie.from(key, value);

        if(domain != null) {
            cookieBuilder.domain(domain);
        }
        if(path != null) {
            cookieBuilder.path(path);
        }
        resp.addCookie(cookieBuilder.build());
    }

    @Override
    protected boolean isPerfTestRequest(ServerWebExchange req) {
        return ReactivePerfTestUtils.isPerfTestRequest(req.getRequest());
    }

    @Override
    protected Map<String, Object> getExPair(ServerWebExchange req) {
        return req.getAttribute(EX_ATTRIBUTE_KEY);
    }

    @Override
    protected Map<String, String> getCookieMap(ServerWebExchange req) {
        Map<String, String> ret = new HashMap<>();
        MultiValueMap<String, HttpCookie> map = req.getRequest().getCookies();

        map.forEach((s, httpCookies) -> {
            ret.put(s, httpCookies.get(httpCookies.size() - 1).getValue());
        });

        return ret;
    }

    @Override
    protected Map<String, String> getLogCookie(ServerWebExchange req) {
        return req.getAttribute(LOG_COOKIE_ATTRIBUTE_KEY);
    }

    @Override
    public void setAppId(ServerWebExchange req, Long appId) {
        req.getAttributes().put(APP_ID_ATTRIBUTE_KEY, appId);
    }

    @Override
    public void setConsumerId(ServerWebExchange req, Long consumerId) {
        req.getAttributes().put(CONSUMER_ID_ATTRIBUTE_KEY, consumerId);
    }

    @Override
    protected Long getAppId(ServerWebExchange req) {
        Long appId = req.getAttribute(APP_ID_ATTRIBUTE_KEY);
        return appId;
    }

    @Override
    protected Long getConsumerId(ServerWebExchange req) {
        Long consumerId = req.getAttribute(CONSUMER_ID_ATTRIBUTE_KEY);
        return consumerId;
    }

    @Override
    public void putLogCookie(ServerWebExchange req, String key, String value) {
        Map<String, String> map = req.getAttribute(LOG_COOKIE_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            req.getAttributes().put(LOG_COOKIE_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }

    @Override
    public void putExPair(ServerWebExchange req, String key, Object value) {
        Map<String, Object> map = req.getAttribute(EX_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            req.getAttributes().put(EX_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }

    @Override
    public void putOverWritePair(ServerWebExchange req, String key, String value) {
        Map<String, String> map = req.getAttribute(OVERWRITE_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            req.getAttributes().put(OVERWRITE_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }
}
