package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duiba.boot.perftest.InternalPerfTestContext;
import cn.com.duiba.boot.perftest.PerfTestDto;
import cn.com.duiba.boot.utils.JarVersionUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import com.netflix.loadbalancer.Server;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * ribbon列表过滤器,此类用于跟压测配合，用于隔离压测流量和正常流量，压测专用服务在注册到eureka中时会附带压测场景ID 和 独立容器集群压测标志，当前服务在发起调用前需要判读如下逻辑：
 * <br/>
 * 1. 如果当前请求带有场景ID && 独立容器集群压测，则过滤eureka服务列表的时候只会选择有相同场景ID的目标服务器进行调用
 * 2. 其余情况，则过滤eureka服务列表的时候只会选择没有场景ID的目标服务器进行调用
 *
 * <br/>
 * 这个类不能加到压测包下，因为压测包下的类只有在引入perftest模块的情况下才会生效，而此类在没有引入perftst模块的情况下也应该生效，否则正常流量可能会调用到压测专用服务器上。
 * <br/>
 * Created by guoyanfei .
 * 2018/12/9 .
 */
public class PerfTestRibbonServerListFilter implements RibbonServerListFilter {

    @Override
    public List<Server> filter(List<Server> serverList, Object key) {   //NOSONAR
        if(serverList.isEmpty()){
            return serverList;
        }
        for(Server s : serverList){
            if(!(s instanceof DiscoveryEnabledServer)){
                return serverList;
            }
        }

        PerfTestDto perfTestLoadBalancerKey = null;
        if (key != null && key instanceof Map) {
            Map<String, Object> loadBalancerInfo = (Map<String, Object>) key;
            perfTestLoadBalancerKey = (PerfTestDto) loadBalancerInfo.get(PerfTestDto.PERF_TEST_DTO_KEY);
        }

        boolean perfTestMode = perfTestLoadBalancerKey != null || InternalPerfTestContext.isCurrentInPerfTestMode();
        String contextSceneId = perfTestLoadBalancerKey != null ? perfTestLoadBalancerKey.getPerfTestSceneId() : InternalPerfTestContext.getCurrentSceneId();
        boolean isTestCluster = perfTestLoadBalancerKey != null ? perfTestLoadBalancerKey.isTestCluster() : InternalPerfTestContext.isTestCluster();

        List<Server> retList = new ArrayList<>();
        for (Server server : serverList) {
            String serverPerfSceneId = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_PERF_SCENE_ID);

            //是压测请求 && 是独立集群压测请求，目标服务器只要和本次请求有相同场景id的目标服务器
            if (perfTestMode && isTestCluster) {
                if (sceneIdEquals(contextSceneId, serverPerfSceneId)) { // 本次请求和目标服务器有相同的场景id
                    String duibaBootVersion = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_BOOT_VERSION);
                    //本机是压测专用服务器，并且与目标服务器的场景ID相同，并且ext版本大于等于1.2.251， 则添加到允许调用的备选服务器列表
                    if (JarVersionUtils.isJarVersionEqualOrGreaterThan(duibaBootVersion, "1.2.251")) {
                        retList.add(server);
                    }
                }
            } else if (StringUtils.isBlank(serverPerfSceneId)) {   // 目标服务器不是压测专用服务器
                retList.add(server);
            }
        }

        //如果当前是压测请求，却找不到被调应用的对应压测实例，直接抛出异常
        if (retList.isEmpty() && perfTestMode) {
            //获得依赖的应用名字
            String dependencyAppName = ((DiscoveryEnabledServer)serverList.get(0)).getInstanceInfo().getAppName();
            if(!"ZIPKIN-SERVER".equals(dependencyAppName)) {//不拦截对ZIPKIN-SERVER的调用。
                throw new IllegalStateException("ribbon过滤服务器时检测到当前请求为带有压测场景ID的请求，但是没有找到" + dependencyAppName + "的压测专用实例(ext>=1.2.234)，拒绝调用。（压测场景ID:" + contextSceneId + "）");
            }
        }

        return retList.isEmpty() ? serverList : retList;
    }

    /**
     * 判断上下文的场景和服务器注册在eureka的场景id是否相同
     * 任何一个为null，都认为不相同
     * @param contextSceneId
     * @param serverSceneId
     * @return
     */
    private boolean sceneIdEquals(String contextSceneId, String serverSceneId) {
        if (contextSceneId == null || serverSceneId == null) {
            return false;
        }
        return contextSceneId.equals(serverSceneId);
    }
}
