package cn.com.duibaboot.ext.autoconfigure.threadpool.wrapper;

import cn.com.duiba.wolf.concurrent.AbortPolicyWithReport;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import cn.com.duibaboot.ext.autoconfigure.threadpool.properties.ThreadPoolProperties;
import cn.com.duibaboot.ext.autoconfigure.threadpool.proxy.ProfileCallable;
import cn.com.duibaboot.ext.autoconfigure.threadpool.proxy.ProfileRunnable;
import com.alibaba.ttl.TtlCallable;
import com.alibaba.ttl.TtlRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.SmartLifecycle;
import org.springframework.core.task.AsyncTaskExecutor;

import javax.annotation.PreDestroy;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.*;

/**
 *
 * @author hwq
 */
public class ThreadPoolExecutorWrapper implements AsyncTaskExecutor,ExecutorService,SmartLifecycle {

    private final ThreadPoolExecutor threadPoolExecutor;

    private final ThreadPoolProperties poolProperties;

    private final String threadPoolName;

    private static final Logger logger = LoggerFactory.getLogger(cn.com.duibaboot.ext.autoconfigure.threadpool.wrapper.ThreadPoolExecutorWrapper.class);

    public ThreadPoolExecutorWrapper(String threadPoolName,ThreadPoolProperties poolProperties){
        this.poolProperties = poolProperties;
        this.threadPoolName = threadPoolName;

        Objects.requireNonNull(threadPoolName);
        Objects.requireNonNull(poolProperties);

        BlockingQueue<Runnable> queue;
        int queueSize = poolProperties.getQueueSize();
        if(queueSize == 0){
            queue = new SynchronousQueue<>();
        }else{
            queue = new ArrayBlockingQueue<>(queueSize);
        }

        int maxSize = Math.max(poolProperties.getMaxSize(), poolProperties.getCoreSize());
        threadPoolExecutor = new ThreadPoolExecutor(poolProperties.getCoreSize(),
                maxSize,
                60L, TimeUnit.SECONDS,
                queue,
                new NamedThreadFactory(threadPoolName),
                new AbortPolicyWithReport());
    }

    @PreDestroy
    @Override
    public void shutdown() {
        threadPoolExecutor.shutdown();
    }
    /**
     * 优雅关闭
     */
    public void shutdownGracefully() {
        //先尝试软关闭线程池
        threadPoolExecutor.shutdown();

        Long shutdownTimeout = poolProperties.getShutdownTimeout();

        try {
            threadPoolExecutor.awaitTermination(shutdownTimeout, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            logger.error("", e);
            Thread.currentThread().interrupt();
        }
        //超时后仍未关闭则强制关闭线程池
        if(!threadPoolExecutor.isShutdown()){
            logger.warn("线程池超过{}ms仍未关闭，强制关闭。", poolProperties.getShutdownTimeout());
            threadPoolExecutor.shutdownNow();
        }
    }

    @Override
    public List<Runnable> shutdownNow() {
        return threadPoolExecutor.shutdownNow();
    }

    @Override
    public boolean isShutdown() {
        return threadPoolExecutor.isShutdown();
    }

    @Override
    public boolean isTerminated() {
        return threadPoolExecutor.isTerminated();
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return threadPoolExecutor.awaitTermination(timeout, unit);
    }

    @Override
    public void execute(Runnable command) {
        threadPoolExecutor.execute(TtlRunnable.get(new ProfileRunnable(command, getQueue(), threadPoolName)));
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
        return threadPoolExecutor.submit(TtlCallable.get(new ProfileCallable(task, getQueue(), threadPoolName)));
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
        return threadPoolExecutor.submit(TtlRunnable.get(new ProfileRunnable(task, getQueue(), threadPoolName)), result);
    }

    @Override
    public void execute(Runnable task, long startTimeout) {
        threadPoolExecutor.execute(TtlRunnable.get(new ProfileRunnable(task, getQueue(), threadPoolName)));
    }

    @Override
    public Future<?> submit(Runnable task) {
        return threadPoolExecutor.submit(TtlRunnable.get(new ProfileRunnable(task, getQueue(), threadPoolName)));
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
        return threadPoolExecutor.invokeAll(TtlCallable.gets(tasks));
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException {
        return threadPoolExecutor.invokeAll(TtlCallable.gets(tasks), timeout, unit);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
        return threadPoolExecutor.invokeAny(TtlCallable.gets(tasks));
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        return threadPoolExecutor.invokeAny(TtlCallable.gets(tasks), timeout, unit);
    }

    public boolean isTerminating() {
        return threadPoolExecutor.isTerminating();
    }

    public BlockingQueue<Runnable> getQueue() {
        return threadPoolExecutor.getQueue();
    }

    public int getPoolSize() {
        return threadPoolExecutor.getPoolSize();
    }

    public int getCorePoolSize() {
        return threadPoolExecutor.getCorePoolSize();
    }

    public int getActiveCount() {
        return threadPoolExecutor.getActiveCount();
    }

    public int getLargestPoolSize() {
        return threadPoolExecutor.getLargestPoolSize();
    }

    public int getMaximumPoolSize() {
        return threadPoolExecutor.getMaximumPoolSize();
    }

    public long getTaskCount() {
        return threadPoolExecutor.getTaskCount();
    }

    public long getCompletedTaskCount() {
        return threadPoolExecutor.getCompletedTaskCount();
    }


    @Override
    public boolean isAutoStartup() {
        return true;
    }

    /**
     * 多个phase相同的SmartLifecycle的stop可以并发执行，所以要在新线程中关闭
     * @param callback
     */
    @Override
    public void stop(final Runnable callback) {
        new Thread(){
            @Override
            public void run(){
                try {
                    ThreadPoolExecutorWrapper.this.shutdownGracefully();
                }finally {
                    callback.run();
                }
            }
        }.start();
    }

    @Override
    public void start() {
        //do nothing
    }

    @Override
    public void stop() {
        //will not be invoked
    }

    @Override
    public boolean isRunning() {
        return !threadPoolExecutor.isTerminating() && !threadPoolExecutor.isTerminated();
    }

    //让GracefulCloseLifeCycle先stop
    @Override
    public int getPhase() {
        return -2;
    }

    public String getThreadPoolName() {
        return threadPoolName;
    }
}
