package cn.com.duibaboot.ext.autoconfigure.cloud.zipkin;

import brave.Tracer;
import brave.sampler.Sampler;
import ch.qos.logback.classic.LoggerContext;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.*;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.cloud.sleuth.instrument.web.TraceWebFilter;
import org.springframework.cloud.sleuth.sampler.SamplerProperties;
import org.springframework.cloud.sleuth.zipkin2.ZipkinAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import javax.servlet.Filter;

/**
 * 设置采样器（最低支持万分之一的采样率）
 */
@Configuration
@ConditionalOnClass({zipkin2.Span.class, ZipkinAutoConfiguration.class})
@EnableConfigurationProperties({SamplerProperties.class})
@AutoConfigureBefore(ZipkinAutoConfiguration.class)
@ConditionalOnProperty(value = "spring.zipkin.enabled", matchIfMissing = true)
public class DuibaZipkinAutoConfiguration {

    @Configuration
    @ConditionalOnClass(RefreshScope.class)
    protected static class RefreshScopedPercentageBasedSamplerConfiguration {
        @Bean
        @RefreshScope
        @ConditionalOnMissingBean
        public Sampler defaultTraceSampler(SamplerProperties config) {
            return new PercentageBasedSampler(config);
        }
    }

    @Configuration
    @ConditionalOnMissingClass("org.springframework.cloud.context.config.annotation.RefreshScope")
    protected static class NonRefreshScopePercentageBasedSamplerConfiguration {
        @Bean
        @ConditionalOnMissingBean
        public Sampler defaultTraceSampler(SamplerProperties config) {
            return new PercentageBasedSampler(config);
        }
    }

    @Configuration
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    @ConditionalOnClass({TraceWebFilter.class, Tracer.class})
    static class ZipkinFiltersConfiguration{
        @Bean
        public Filter duibaTraceFilter() {
            LoggerContext context = (LoggerContext)LoggerFactory.getILoggerFactory();
            context.addTurboFilter(new ZipkinLogTurboFilter());

            return new DuibaTraceFilter();
        }

        @Bean
        public Filter duibaTracSecondFilter() {
            return new DuibaTraceSecondFilter();
        }

        @Bean
        public DuibaRpcParamsInterceptor duibaRpcParamsInterceptor() {
            return new DuibaRpcParamsInterceptor();
        }

        @Bean
        public ZipkinWebMvcConfigurer zipkinWebMvcConfigurer(){
            return new ZipkinWebMvcConfigurer();
        }
    }

    static class ZipkinWebMvcConfigurer implements WebMvcConfigurer{

        @Autowired
        private DuibaRpcParamsInterceptor duibaRpcParamsInterceptor;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            registry.addInterceptor(duibaRpcParamsInterceptor).addPathPatterns("/**");
        }
    }

    @Bean
    @ConditionalOnProperty(value = "duiba.zipkin.port", matchIfMissing = false)
    public static SpecifiedBeanPostProcessor zipkinLoadBalancerBeanPostProcessor(@Value("${duiba.zipkin.port}") int zipkinPort) {
        return new ZipkinLoadBalancerBeanPostProcessor(zipkinPort);
    }
}
