package cn.com.duibaboot.ext.autoconfigure.etcd.client;

import cn.com.duiba.wolf.utils.SecurityUtils;
import cn.com.duibaboot.ext.autoconfigure.etcd.exception.EtcdOperationException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.Charset;
import java.util.*;

/**
 * 封装jetcd的kvClient 提供etcd简单的 put get getWithPrefix delete deleteWithPrefix 操作
 *
 * rest版本
 * 由于jetcd依赖到了3.5.1版本的protobuf，而hbase依赖2.5.0版本的protobuf，有jar包冲突的问题（用shade插件也不好解决，因为protobuf包里有很多资源文件）
 * 所以部分依赖了hbase的项目，我们允许他们exclude掉jetcd的jar包，这样的话我们就需要自己封装一个简单版本的rest etcd客户端。
 *
 * 2018/7/12 .
 */
public class RestEtcdKVClientDelegate implements EtcdKVClientDelegate {

    private static final Logger logger = LoggerFactory.getLogger(RestEtcdKVClientDelegate.class);

    private RestTemplate restTemplate;
    private String[] etcdUris;

    //TODO 如果etcd服务器升级版本，这个API_VERSION有可能要改
    private String curApiVersion = "/v3alpha";
    private static final String[] API_VERSIONS = {"/v3alpha", "/v3beta"};
    private static final int MAX_RETRY_TIMES = 2;

    public RestEtcdKVClientDelegate(String... etcdUris) {
        restTemplate = new RestTemplateBuilder().setConnectTimeout(5000).setReadTimeout(5000)
                .build();

        this.etcdUris = etcdUris;
        if(etcdUris == null || etcdUris.length == 0){
            throw new IllegalArgumentException("etcdUris must not be empty");
        }

        //探测API_VERSION
        for(String apiVersion : API_VERSIONS) {
            for(String etcdUri : etcdUris) {
                try {
                    ResponseEntity<String> resp = restTemplate.postForEntity(etcdUri + apiVersion + "/kv/range", "{\"key\": \"Zm9v\"}", String.class);
                    if(resp.getStatusCodeValue() == 200){
                        this.curApiVersion = apiVersion;
                        return;
                    }
                }catch (Throwable e){
                    //Ignore
                    logger.info("", e);
                }
            }
        }
        throw new IllegalStateException("无法探测etcd的http api版本号, etcdUris: " + Arrays.toString(etcdUris));
    }

//        curl -L http://172.16.80.120:2379/v3alpha/kv/put \
//        -X POST -d '{"key": "Zm9v", "value": "YmFy111"}'
//        curl -L http://172.16.80.120:2379/v3alpha/kv/deleterange \
//        -X POST -d '{"key": "Zm9v", "range_end": "Zm9w"}'
//        curl -L http://172.16.80.120:2379/v3alpha/kv/range \
//        -X POST -d '{"key": "Zm9v", "range_end": "Zm9w"}'
    public void put(String key, String value) {
        JSONObject jsonObject = new JSONObject();

        jsonObject.put("key", SecurityUtils.encode2StringByBase64(key.getBytes()));
        jsonObject.put("value", SecurityUtils.encode2StringByBase64(value.getBytes()));

        postForObject("/kv/put", jsonObject);
    }

    private JSONObject postForObject(String path, JSONObject jsonBody){
        for(int i=0;i<=MAX_RETRY_TIMES;i++) {
            try {
                String uri = etcdUris[RandomUtils.nextInt(0, etcdUris.length)];
                String result = restTemplate.postForObject(uri + curApiVersion + path, jsonBody.toString(), String.class);
                JSONObject resultJson = JSON.parseObject(result);
                if (resultJson.containsKey("error")) {
                    throw new EtcdOperationException("操作etcd时发生错误, 错误响应：" + result);
                }
                return resultJson;
            } catch (Throwable e) {
                if(i < MAX_RETRY_TIMES) {
                    logger.warn("操作etcd时发生错误，将自动重试,第{}/{}次重试", i+1, MAX_RETRY_TIMES, e);
                    continue;
                }else{
                    throw new EtcdOperationException("操作etcd重试"+MAX_RETRY_TIMES+"次后仍然失败", e);
                }
            }
        }

        throw new EtcdOperationException("[NOTIFYME]操作etcd时发生错误, 不应该运行到这里");
    }

    public String get(String key) {
        JSONObject jsonObject = new JSONObject();

        jsonObject.put("key", SecurityUtils.encode2StringByBase64(key.getBytes()));

        JSONObject resultJson = postForObject("/kv/range", jsonObject);

        JSONArray arr = resultJson.getJSONArray("kvs");
        if(arr != null && !arr.isEmpty()){
            JSONObject obj = arr.getJSONObject(0);
            return new String(SecurityUtils.decodeBase64(StringUtils.defaultString(obj.getString("value"))), Charset.forName("utf-8"));
        }
        return null;
    }

    public Map<String, String> getWithPrefix(String prefix) {
        JSONObject jsonObject = new JSONObject();

        jsonObject.put("key", SecurityUtils.encode2StringByBase64(prefix.getBytes()));
        jsonObject.put("range_end", SecurityUtils.encode2StringByBase64(prefixEndOf(prefix.getBytes())));

        JSONObject resultJson = postForObject("/kv/range", jsonObject);

        JSONArray arr = resultJson.getJSONArray("kvs");
        Map<String,String> map = new LinkedHashMap<>();
        if(arr != null && !arr.isEmpty()){
            for(int i=0;i<arr.size();i++) {
                JSONObject obj = arr.getJSONObject(i);
                String key = new String(SecurityUtils.decodeBase64(obj.getString("key")), Charset.forName("utf-8"));
                String value = new String(SecurityUtils.decodeBase64(StringUtils.defaultString(obj.getString("value"))), Charset.forName("utf-8"));
                map.put(key, value);
            }
        }
        return map;
    }

    public void delete(String key) {
        JSONObject jsonObject = new JSONObject();

        jsonObject.put("key", SecurityUtils.encode2StringByBase64(key.getBytes()));

        postForObject("/kv/deleterange", jsonObject);
    }

    public void deleteWithPrefix(String prefix) {
        JSONObject jsonObject = new JSONObject();

        jsonObject.put("key", SecurityUtils.encode2StringByBase64(prefix.getBytes()));
        jsonObject.put("range_end", SecurityUtils.encode2StringByBase64(prefixEndOf(prefix.getBytes())));

        postForObject("/kv/deleterange", jsonObject);
    }

    private final byte[] prefixEndOf(byte[] prefix) {
        byte[] endKey = prefix.clone();
        for (int i = endKey.length - 1; i >= 0; i--) {
            if (endKey[i] < 0xff) {
                endKey[i] = (byte) (endKey[i] + 1);
                return Arrays.copyOf(endKey, i + 1);
            }
        }

        return new byte[]{0};
    }

    public static void main(String[] args) {
        RestEtcdKVClientDelegate delegate = new RestEtcdKVClientDelegate("http://172.16.80.120:2379");//NOSONAR
        delegate.put("/config1/zuul-server/a","fuckit");
        delegate.put("/config1/zuul-server/b","fuckit");
        delegate.put("/config1/zuul-server/c","fuckit");
        String value = delegate.get("/config1/zuul-server/a");
        System.out.println(value);
        Map<String,String> map = delegate.getWithPrefix("/config1/");
        System.out.println(map);
        delegate.delete("/config1/zuul-server/b");
        delegate.deleteWithPrefix("/config1/zuul-server/");
        map = delegate.getWithPrefix("/config1");
        System.out.println(map);
    }

}
