package cn.com.duibaboot.ext.autoconfigure.cat;

import brave.Span;
import brave.Tags;
import brave.Tracer;
import brave.Tracing;
import io.lettuce.core.RedisFuture;
import io.lettuce.core.cluster.api.async.RedisClusterAsyncCommands;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.data.redis.connection.RedisConnection;

/**
 * 加入aop，监控spring-data-redis执行耗时
 */
@Aspect
@Order(-1)
public class SleuthSpringDataLettucePlugin {

	private static final Logger logger = LoggerFactory.getLogger(SleuthSpringDataLettucePlugin.class);

	@Autowired
	private Tracer tracer;
	@Autowired
	private Tracing tracing;

	private volatile Class<?> lastRedisConnectionProxyClass;

	@Around("execution(* org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory.getConnection(..))")
	public Object springDataLettuceJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable{
		Span curSpan = tracer.currentSpan();
		if(curSpan == null || curSpan.isNoop()){
			return joinPoint.proceed();
		}

		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		String methodName = signature.getMethod().getName();

		if("getConnection".equals(methodName)){
			try {
				RedisConnection connection = (RedisConnection) joinPoint.proceed();

				ProxyFactory factory = new ProxyFactory();
				factory.setTarget(connection);
				//factory.setTargetClass(JedisConnection.class);
				factory.addAdvice(new RedisConnectionMethodInterceptor());
				RedisConnection newConnection = (RedisConnection) factory.getProxy();//getClass().getClassLoader()

				if(lastRedisConnectionProxyClass != null && lastRedisConnectionProxyClass != newConnection.getClass()){
					logger.error("JedisConnectionProxyClass is not same，this is spring's bug,please upgrade spring-boot's version to 1.3.8.RELEASE or higher! {},{},{},{}",new Object[]{getClass().getClassLoader(), newConnection.getClass().getClassLoader(), lastRedisConnectionProxyClass, newConnection.getClass()});//如果抱这个错，则有perm区内存溢出的风险，需要关注
				}
				lastRedisConnectionProxyClass = newConnection.getClass();

				return newConnection;
			} catch (Throwable e) {
				//失败才记录
				Span span = tracer.nextSpan().name("redis:/"+methodName).kind(Span.Kind.CLIENT)
						.remoteServiceName("redis")
						.start();
				if(!span.isNoop()) {
					// get/...
					span.tag("redis.op", methodName);
					//本地组件名
					span.tag("lc", "springDataLettuce");
					Tags.ERROR.tag(e,span);
				}
				span.finish();

				throw e;
			}
		}else{
			return joinPoint.proceed();
		}
	}

	/**
	 * {@link MethodInterceptor}
	 */
	private class RedisConnectionMethodInterceptor implements MethodInterceptor {

		RedisConnectionMethodInterceptor() {
		}

		@Override
		public Object invoke(MethodInvocation invocation) throws Throwable {
			String methodName = invocation.getMethod().getName();

			if(methodName.equals("isPipelined") || methodName.equals("openPipeline") || methodName.equals("isQueueing")
					|| methodName.equals("isClosed")
					|| methodName.equals("close")
					|| methodName.equals("closePipeline")
					|| methodName.endsWith("Commands")){
				return invocation.proceed();
			}else if(methodName.equals("getNativeConnection")){
				Object nativeConnection = invocation.proceed();

				Object newNativeConnection = nativeConnection;
				ProxyFactory factory = new ProxyFactory();
				factory.setTarget(nativeConnection);
				if(nativeConnection instanceof RedisClusterAsyncCommands){//异步
					factory.addAdvice(new LettuceAsyncMethodInterceptor());
					newNativeConnection = factory.getProxy();
				}

				return newNativeConnection;
			}

			return interceptorRedisMethods(invocation);
		}

	}

	/**
	 * {@link MethodInterceptor}
	 */
	private class LettuceAsyncMethodInterceptor implements MethodInterceptor {

		@Override
		public Object invoke(MethodInvocation invocation) throws Throwable {	// NOSONAR
			if(!invocation.getMethod().getReturnType().equals(RedisFuture.class)) {
				return invocation.proceed();
			}
			String methodName = invocation.getMethod().getName();

            Span curSpan = tracer.currentSpan();
            if(curSpan == null || curSpan.isNoop()){
                return invocation.proceed();
            }

            long startTimeMillis = tracing.clock(curSpan.context()).currentTimeMicroseconds();

			RedisFuture future = (RedisFuture)invocation.proceed();
			future.whenComplete((s, throwable) -> {
			    try(Tracer.SpanInScope ws = tracer.withSpanInScope(curSpan)) {
			        Span span = tracer.nextSpan().name("redis:/" + methodName).kind(Span.Kind.CLIENT)
                            .remoteServiceName("redis")
                            .start(startTimeMillis);
			        if(!span.isNoop()) {
						span.tag("redis.op", methodName);// get/...
						span.tag("lc", "springDataLettuce");//本地组件名
						if ("get".equals(methodName)
								&& invocation.getArguments() != null
								&& invocation.getArguments().length == 1
								&& invocation.getArguments()[0] != null) {
							Object obj = invocation.getArguments()[0];
							if (obj instanceof byte[]) {
								span.tag("redis.key", new String((byte[]) obj));//key
							}
							else {
								span.tag("redis.key", obj.toString());//key
							}
						}

						if (throwable != null) {
							Tags.ERROR.tag((Throwable)throwable,span);
						}
					}
                    span.finish();
                }
			});
			return future;
		}

	}

	private Object interceptorRedisMethods(MethodInvocation invocation) throws Throwable {
		String methodName = invocation.getMethod().getName();
		//忽略Object基类中的方法
		if(methodName.equals("toString") || methodName.equals("hashCode") || methodName.equals("equals")){
			return invocation.proceed();
		}

		Span span = tracer.nextSpan().name("redis:/"+methodName).kind(Span.Kind.CLIENT)
				.remoteServiceName("redis")
				.start();
		try(Tracer.SpanInScope scope = tracer.withSpanInScope(span)) {
			if(!span.isNoop()) {
				span.tag("redis.op", methodName);// get/...
				span.tag("lc", "springDataLettuce");//本地组件名
				if ("get".equals(methodName)
						&& invocation.getArguments() != null
						&& invocation.getArguments().length == 1
						&& invocation.getArguments()[0] != null) {
					Object obj = invocation.getArguments()[0];
					if (obj instanceof byte[]) {
						span.tag("redis.key", new String((byte[]) obj));//key
					}
					else {
						span.tag("redis.key", obj.toString());//key
					}
				}
			}
//			span.logEvent(Span.CLIENT_SEND);

			return invocation.proceed();
		} catch(Exception e){
			Tags.ERROR.tag(e,span);
			throw e;
		} finally {
//			span.logEvent(Span.CLIENT_RECV);
			span.finish();
		}
	}

}
