package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.hystrix;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.DuibaFeignProperties;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duiba.boot.netflix.feign.hystrix.FeignHystrixCommand;
import cn.com.duiba.boot.netflix.feign.hystrix.HystrixDisabled;
import com.alibaba.fastjson.JSONObject;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import feign.FeignException;
import feign.Response;
import feign.Util;
import feign.codec.ErrorDecoder;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;

import java.io.IOException;
import java.lang.reflect.Method;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.RejectedExecutionException;

import static feign.Util.RETRY_AFTER;
import static feign.Util.checkNotNull;
import static java.lang.String.format;
import static java.util.Locale.US;
import static java.util.concurrent.TimeUnit.SECONDS;

/**
 * 自定义Feign错误解码器，把非2**状态的http响应转换为异常。对于IllegalArgumentException或者注解了@HystrixDisabled的异常会包装为HystrixBadRequestException，避免触发熔断/降级
 */
public class FeignErrorDecoder implements ErrorDecoder {

    private static final Logger logger = LoggerFactory.getLogger(FeignErrorDecoder.class);

//    private final RetryAfterDecoder retryAfterDecoder = new RetryAfterDecoder();
    private final ErrorDecoder.Default defaultErrorDecoder = new Default();

    @Override
    public Exception decode(String methodKey, Response response) {
        Exception exception;
        try {
            exception = decodeInner(methodKey, response);
        } catch (IOException ignored) {
            exception = genFeignException(response.status(), methodKey, null);
        }

        if(exception instanceof FeignException) {
            return defaultErrorDecoder.decode(methodKey, response);
        }

        return exception;
    }

    private Exception decodeInner(String methodKey, Response response) throws IOException {
        Exception exception;
        if (response.body() != null) {
            JSONObject body = getBodyString(response);
            String exceptionClassStr = body.getString("exception");
            Class<?> exceptionClass = tryGetExceptionClass(exceptionClassStr);

            if(exceptionClass != null){
                if(BizException.class.isAssignableFrom(exceptionClass)){//BizException特殊处理
                    exception = new BizException(body.getString("message")).withCode(body.getString("code"));
                }else {
                    exception = genFeignException(response.status(), methodKey, body.toJSONString());
                }

                if(IllegalArgumentException.class.isAssignableFrom(exceptionClass)
                        || exceptionClass.isAnnotationPresent(HystrixDisabled.class)
                        || isInMethodIgnoreExceptions(RpcContext.getContext().getMethod(), exceptionClass)){
                    //不允许hystrix进行熔断、降级的异常用HystrixBadRequestException包装起来。
                    exception = new HystrixBadRequestException(exception.getMessage(), exception);
                }else if(RejectedExecutionException.class.isAssignableFrom(exceptionClass)){//服务端拒绝执行,可能是线程池满了/服务器忙
//                    exception = new RetryableException(response.status(), exception.getMessage(),response.request().httpMethod(), new RejectedExecutionException(exception.getMessage(), exception), null);
                }
            }else{
                exception = genFeignException(response.status(), methodKey, body.toJSONString());
            }
        }else{
            exception = genFeignException(response.status(), methodKey, null);
        }

        return exception;
    }

    private Class tryGetExceptionClass(String exceptionClassStr){
        Class<?> exceptionClass = null;
        if(exceptionClassStr != null){
            try{
                exceptionClass = Class.forName(exceptionClassStr);
            } catch (ClassNotFoundException e) {
                //Ignore
//                        logger.warn("exceptionClass [{}] not found", exceptionClass);
            }
        }

        return exceptionClass;
    }

    /**
     * 解析body数据，body数据可能是fst/kryo/json序列化的。
     * @param response
     * @return
     * @throws IOException
     */
    private JSONObject getBodyString(Response response) throws IOException {
        String contentType = null;
        Collection<String> coll = response.headers().get(HttpHeaders.CONTENT_TYPE.toLowerCase());//Feign会把所有headerName转为小写
        if(coll != null && !coll.isEmpty()){
            contentType = coll.iterator().next();
        }

        DuibaFeignProperties.DuibaFeignSerialization stype = DuibaFeignProperties.DuibaFeignSerialization.contentTypeOf(contentType);

        JSONObject body;
        if (stype == null || stype == DuibaFeignProperties.DuibaFeignSerialization.JSON) {
            body = JSONObject.parseObject(Util.toString(response.body().asReader()));
        } else {
            byte[] bs = IOUtils.toByteArray(response.body().asInputStream());
            Object obj = stype.deserialize(bs);
            if(obj instanceof String){
                body = JSONObject.parseObject((String)obj);
            }else{
                body = (JSONObject)JSONObject.toJSON(obj);
            }
        }
        return body;
    }

    /**
     * 判断异常是否在熔断器忽略异常列表中.
     * @param method
     * @param exceptionClass
     * @return
     */
    private boolean isInMethodIgnoreExceptions(Method method, Class exceptionClass){
        FeignHystrixCommand commandOfMethod = method.getAnnotation(FeignHystrixCommand.class);
        FeignHystrixCommand commandOfClass = method.getDeclaringClass().getAnnotation(FeignHystrixCommand.class);
        return isInIgnoreExceptions(commandOfMethod, exceptionClass)
                || isInIgnoreExceptions(commandOfClass, exceptionClass);
    }

    private boolean isInIgnoreExceptions(FeignHystrixCommand command, Class exceptionClass){
        if(command != null){
            Class[] classes = command.ignoreExceptions();
            if(classes != null){
                for(Class clazz : classes){
                    if(clazz.isAssignableFrom(exceptionClass)){
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private <T> T firstOrNull(Map<String, Collection<T>> map, String key) {
        if (map.containsKey(key) && !map.get(key).isEmpty()) {
            return map.get(key).iterator().next();
        }
        return null;
    }

    //组装的exception会带上本次Feign调用的本机地址端口和远程地址端口
    private FeignClientException genFeignException(int status, String methodKey, String responseBody){
        StringBuilder sb = new StringBuilder();
        if(RpcContext.hasContext()) {
            RpcContext rc = RpcContext.getContext();
            sb.append("(").append(rc.getLocalAddr()).append(" -> ").append(rc.getRemoteAddr()).append(")");
        }

        sb.append(format("status %s reading %s", status, methodKey));
        sb.append("; content:\n").append(responseBody);
        return new FeignClientException(status, sb.toString());
    }

    /**
     * Decodes a {@link Util#RETRY_AFTER} header into an absolute date, if possible. <br> See <a
     * href="https://tools.ietf.org/html/rfc2616#section-14.37">Retry-After format</a>
     */
    static class RetryAfterDecoder {

        static final DateFormat
                RFC822_FORMAT =
                new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss 'GMT'", US);
        private final DateFormat rfc822Format;

        RetryAfterDecoder() {
            this(RFC822_FORMAT);
        }

        RetryAfterDecoder(DateFormat rfc822Format) {
            this.rfc822Format = checkNotNull(rfc822Format, "rfc822Format");
        }

        protected long currentTimeMillis() {
            return System.currentTimeMillis();
        }

        /**
         * returns a date that corresponds to the first time a request can be retried.
         *
         * @param retryAfter String in <a href="https://tools.ietf.org/html/rfc2616#section-14.37"
         *                   >Retry-After format</a>
         */
        public Date apply(String retryAfter) {
            if (retryAfter == null) {
                return null;
            }
            if (retryAfter.matches("^[0-9]+$")) {
                long deltaMillis = SECONDS.toMillis(Long.parseLong(retryAfter));
                return new Date(currentTimeMillis() + deltaMillis);
            }
            synchronized (rfc822Format) {
                try {
                    return rfc822Format.parse(retryAfter);
                } catch (ParseException ignored) {
                    return null;
                }
            }
        }
    }

}
