/*
 * Copyright 2012-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.duibaboot.ext.autoconfigure.web;

import cn.com.duibaboot.ext.autoconfigure.core.DuibaServerProperties;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.web.container.DuibaTomcatContextCustomizer;
import cn.com.duibaboot.ext.autoconfigure.web.container.JettyCustomizer;
import cn.com.duibaboot.ext.autoconfigure.web.container.TomcatCustomizer;
import cn.com.duibaboot.ext.autoconfigure.web.container.UndertowCustomizer;
import io.undertow.Undertow;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.util.Loader;
import org.eclipse.jetty.webapp.WebAppContext;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.*;
import org.springframework.boot.autoconfigure.web.servlet.ServletWebServerFactoryAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.embedded.jetty.JettyServletWebServerFactory;
import org.springframework.boot.web.embedded.tomcat.TomcatServletWebServerFactory;
import org.springframework.boot.web.embedded.undertow.UndertowServletWebServerFactory;
import org.springframework.boot.web.server.ConfigurableWebServerFactory;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.boot.web.servlet.server.ServletWebServerFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.xnio.SslClientAuthMode;

import javax.annotation.Resource;
import javax.servlet.DispatcherType;
import javax.servlet.Servlet;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;

/**
 * 自动配置Tomcat、Jetty、Undertow等容器，增加http线程池监控，如果在队列中等待超过1秒，则打印error日志提示线程池过小
 * <br/>
 * 如果是内部服务，使用固定大小的线程池,并拒绝在http线程池工作队列中等待过久的任务，让客户端可以尽快重试
 */
@Configuration
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
@EnableConfigurationProperties({DuibaServerProperties.class})
@AutoConfigureBefore(ServletWebServerFactoryAutoConfiguration.class)
public class DuibaEmbeddedServletContainerAutoConfiguration {

    //如下两个Embed*是从EmbeddedServletContainerAutoConfiguration复制过来的，作用是当同时引入了tomcat和jetty依赖时优先使用jetty，当同时引入了tomcat和undertow时优先使用undertow.
    /**
     * Nested configuration if Jetty is being used.
     */
    @Configuration
    @ConditionalOnClass({ Servlet.class, Server.class, Loader.class,
            WebAppContext.class })
    @ConditionalOnMissingBean(value = ServletWebServerFactory.class, search = SearchStrategy.CURRENT)
    public static class EmbeddedJetty {

        @Bean
        public JettyServletWebServerFactory jettyEmbeddedServletContainerFactory() {
            return new JettyServletWebServerFactory();
        }

    }

    /**
     * Nested configuration if Undertow is being used.
     */
    @Configuration
    @ConditionalOnClass({ Servlet.class, Undertow.class, SslClientAuthMode.class })
    @ConditionalOnMissingBean(value = ServletWebServerFactory.class, search = SearchStrategy.CURRENT)
    public static class EmbeddedUndertow {

        @Bean
        public UndertowServletWebServerFactory undertowEmbeddedServletContainerFactory() {
            return new UndertowServletWebServerFactory();
        }

    }

	@Resource
	private DuibaServerProperties duibaServerProperties;

	/**
	 * 自动配置Tomcat、Jetty等容器，增加http线程池监控，如果在队列中等待超过1秒，则打印error日志提示线程池过小<br/>
	 * 如果是内部服务，使用固定大小的线程池,并拒绝在http线程池工作队列中等待过久的任务，让客户端可以尽快重试
	 * @return
	 */
	@Bean
	//@ConditionalOnBean(TomcatServletWebServerFactory.class) 这行代码不能加，否则不会生效
	public TomcatCustomizer tomcatCustomizer(){
		return new TomcatCustomizer(duibaServerProperties);
	}

	/**
	 * @return
	 * @Bean //todo:因为线程池类型变更，暂时不进行监听
	 */
	@ConditionalOnBean(UndertowServletWebServerFactory.class)
	public UndertowCustomizer undertowCustomizer(){
		return new UndertowCustomizer(duibaServerProperties);
	}

	@Bean
	@ConditionalOnBean(JettyServletWebServerFactory.class)
	public JettyCustomizer jettyCustomizer(){
		return new JettyCustomizer();
	}

	@Bean
	public static SpecifiedBeanPostProcessor servletContainerBeanPostProcessor(){
		return new SpecifiedBeanPostProcessor<ConfigurableWebServerFactory>() {
			@Override
			public int getOrder() {
				return 0;
			}

			@Override
			public Class<ConfigurableWebServerFactory> getBeanType() {
				return ConfigurableWebServerFactory.class;
			}

			@Override
			public Object postProcessBeforeInitialization(ConfigurableWebServerFactory bean, String beanName) {
				return bean;
			}

			@Override
			public Object postProcessAfterInitialization(ConfigurableWebServerFactory bean, String beanName) {
				if(bean instanceof TomcatServletWebServerFactory){
					TomcatServletWebServerFactory tb = (TomcatServletWebServerFactory)bean;
					tb.addContextCustomizers(new DuibaTomcatContextCustomizer());
				}
				return bean;
			}
		};
	}

	/**
	 * FailFastFilter，判断当前http线程在 httpFailFastExecutor（线程名前缀为HttpFailFastThread）中执行时，直接抛出RejectedExecutionException
	 * ，以让HTTP Rest RPC客户端尽快到其他客户端安全地重试。
	 * @return
	 */
	@Bean
	public FilterRegistrationBean getFailFastFilter(){
		FailFastFilter filter=new FailFastFilter();
		FilterRegistrationBean registrationBean = new FilterRegistrationBean();
		registrationBean.setFilter(filter);
		List<String> urlPatterns=new ArrayList<>();
		urlPatterns.add("/*");//拦截路径，可以添加多个
		registrationBean.setUrlPatterns(urlPatterns);
		registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
		registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 1);//最高优先级(但是要在OrderedCharacterEncodingFilter后面)
		return registrationBean;
	}

}
