package cn.com.duiba.spring.boot.starter.dsp.util;

import com.google.common.base.Joiner;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author wangwei
 * @since 2021/12/7 11:10 上午
 */
public class RedisBloomHandler<K, V> {

    private static final String RESERVE = "BF.RESERVE";
    private static final String ADD = "BF.ADD";
    private static final String MADD = "BF.MADD";
    private static final String EXISTS = "BF.EXISTS";
    private static final String MEXISTS = "BF.MEXISTS";

    @Resource(name = "redisBloomRedisTemplate")
    private RedisTemplate redisTemplate;

    RedisSerializer keySerializer() {
        return redisTemplate.getKeySerializer();
    }

    RedisSerializer valueSerializer() {
        return redisTemplate.getValueSerializer();
    }

    RedisSerializer hashKeySerializer() {
        return redisTemplate.getHashKeySerializer();
    }

    RedisSerializer hashValueSerializer() {
        return redisTemplate.getHashValueSerializer();
    }

    RedisSerializer stringSerializer() {
        return redisTemplate.getStringSerializer();
    }

    public void createFilter(K key, double errorRate, long initCapacity) {
        byte[] rawKey = rawKey(key);
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
            return null;
        }, true);
    }

    public void createFilters(List<K> keys, double errorRate, long initCapacity) {
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            keys.forEach(key -> {
                byte[] rawKey = rawKey(key);
                connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
            });
            return null;
        }, true);
    }

    public void createFilterWithExpire(K key, double errorRate, long initCapacity, long seconds) {
        byte[] rawKey = rawKey(key);
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
            connection.expire(rawKey, seconds);
            return null;
        }, true);
    }

    public void createFiltersWithExpire(List<K> keys, double errorRate, long initCapacity, long seconds) {
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            keys.forEach(key -> {
                byte[] rawKey = rawKey(key);
                connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
                connection.expire(rawKey, seconds);
            });
            return null;
        }, true);
    }

    public Boolean add(K key, V value) {
        byte[] rawKey = rawKey(key);
        byte[] rawValue = rawValue(value);
        return (Boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(ADD, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public Boolean[] addMulti(K key, V... values) {
        byte[][] rawArgs = rawArgs(key, values);
        return (Boolean[]) redisTemplate.execute(connection -> {
            List<Long> ls = (List<Long>) connection.execute(MADD, rawArgs);
            return ls.stream().map(l -> Objects.equals(l, 1L)).toArray(Boolean[]::new);
        }, true);
    }

    public boolean exists(K key, V value) {
        byte[] rawKey = rawKey(key);
        byte[] rawValue = rawValue(value);
        return (boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(EXISTS, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public Boolean[] existsMulti(K key, V... values) {
        byte[][] rawArgs = rawArgs(key, values);
        return (Boolean[]) redisTemplate.execute(connection -> {
            List<Long> ls = (List<Long>) connection.execute(MEXISTS, rawArgs);
            return ls.stream().map(l -> Objects.equals(l, 1L)).toArray(Boolean[]::new);
        }, true);
    }

    public Boolean delete(K key) {
        return redisTemplate.delete(key);
    }

    public Boolean expire(K key, long timeOut, TimeUnit timeUnit) {
        return redisTemplate.expire(key, timeOut, timeUnit);
    }

    public Boolean hasBloom(K key) {
        return redisTemplate.hasKey(key);
    }

    byte[] rawKey(Object key) {
        Assert.notNull(key, "non null key required");
        return this.keySerializer() == null && key instanceof byte[] ? (byte[]) ((byte[]) key) : this.keySerializer().serialize(key);
    }

    byte[] rawString(String key) {
        return this.stringSerializer().serialize(key);
    }

    byte[] rawValue(Object value) {
        return this.valueSerializer() == null && value instanceof byte[] ? (byte[]) ((byte[]) value) : this.valueSerializer().serialize(value);
    }

    private byte[][] rawArgs(Object key, Object... values) {
        byte[][] rawArgs = new byte[1 + values.length][];

        int i = 0;
        rawArgs[i++] = rawKey(key);

        for (Object value : values) {
            rawArgs[i++] = rawValue(value);
        }

        return rawArgs;
    }

    /**
     * 判断多个key是否存在，返回不存在的key
     *
     * @param keys
     * @return
     */
    public List<String> hasKeys(List<String> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return null;
        }
        List<String> filterList = new ArrayList<>();
        redisTemplate.executePipelined((RedisCallback<String>) connection -> {
            for (String key : keys) {
                byte[] rawKey = rawKey(key);
                Boolean exists = connection.exists(rawKey);
                if (exists == null || !exists) {
                    filterList.add(key);
                }
            }
            return null;
        });
        return filterList;
    }

    public Integer saveBloom(String key, String value, Integer levelCount) {
        final Integer[] level = {null};
        redisTemplate.executePipelined((RedisCallback<Integer>) connection -> {
            for (Integer i = 0; i < levelCount; i++) {
                String newValue = value + "_" + i;
                byte[] rawKey = rawKey(key);
                byte[] rawValue = rawValue(newValue);
                Long l = (Long) connection.execute(EXISTS, rawKey, rawValue);
                if (Objects.equals(l, 0L)) {
                    connection.execute(ADD, rawKey, rawValue);
                    level[0] = i;
                    break;
                }
            }
            return null;
        });
        return level[0];
    }

    public void createFilterWithExpireAt(K key, double errorRate, long initCapacity, long expireAt) {
        byte[] rawKey = rawKey(key);
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
            connection.expireAt(rawKey, expireAt);
            return null;
        }, true);
    }
}
