package cn.com.duibaboot.ext.autoconfigure.dubbo.cat;

import cn.com.duiba.cat.Cat;
import cn.com.duiba.cat.message.Event;
import cn.com.duiba.cat.message.Message;
import cn.com.duiba.cat.message.Transaction;
import cn.com.duiba.cat.message.internal.AbstractMessage;
import cn.com.duibaboot.ext.autoconfigure.cat.context.CatConstants;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.metadata.MetadataService;
import org.apache.dubbo.rpc.*;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by houwen
 */
@Activate(group = {CommonConstants.PROVIDER, CommonConstants.CONSUMER}, order = -9000)
public class DubboCatTransaction implements Filter {

    private static final ThreadLocal<Cat.Context> CAT_CONTEXT = new ThreadLocal<>();

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        if (!CatUtils.isCatEnabled()) {
            return invoker.invoke(invocation);
        }
        if (invoker.getInterface().equals(MetadataService.class)) {
            return invoker.invoke(invocation);
        }
        URL url = invoker.getUrl();
        String sideKey = url.getParameter(CommonConstants.SIDE_KEY);
        String loggerName = invoker.getInterface().getSimpleName() + "." + invocation.getMethodName();
        String type = CatConstants.CROSS_CONSUMER;
        if (CommonConstants.PROVIDER_SIDE.equals(sideKey)) {
            type = CatConstants.CROSS_SERVER;
        }
        Transaction transaction = Cat.newTransaction(type, loggerName);
        Result result;
        try {
            Cat.Context context = getContext();
            if (CommonConstants.CONSUMER_SIDE.equals(sideKey)) {
                createConsumerCross(url, transaction);
                Cat.logRemoteCallClient(context);
            } else {
                createProviderCross(transaction);
                Cat.logRemoteCallServer(context);
            }
            setAttachment(context);

            result = invoker.invoke(invocation);

            if (result.hasException()) {
                transaction.setStatus(result.getException());
            } else {
                transaction.setStatus(Message.SUCCESS);
            }
        } catch (Exception e) {
            if (e instanceof RpcException && e.getCause() != null && e.getCause() instanceof InterruptedException) {
                transaction.setStatus(e.getClass().getSimpleName() + ":Caused By:InterruptedException");
            } else {
                transaction.setStatus(e);
            }
            throw e;
        } finally {
            transaction.complete();
            clearContext();
        }
        return result;
    }

    static class DubboCatContext implements Cat.Context {

        private Map<String, String> properties = new HashMap<String, String>();

        @Override
        public void addProperty(String key, String value) {
            properties.put(key, value);
        }

        @Override
        public String getProperty(String key) {
            return properties.get(key);
        }
    }

    private String getProviderAppName(URL url) {
        String appName = url.getParameter(CatConstants.PROVIDER_APPLICATION_NAME);
        if (StringUtils.isEmpty(appName)) {
            String interfaceName = url.getParameter(CommonConstants.INTERFACE_KEY);
            appName = interfaceName.substring(0, interfaceName.lastIndexOf('.'));
        }
        return appName;
    }

    private void setAttachment(Cat.Context context) {
        RpcContext.getContext().setAttachment(Cat.Context.ROOT, context.getProperty(Cat.Context.ROOT));
        RpcContext.getContext().setAttachment(Cat.Context.CHILD, context.getProperty(Cat.Context.CHILD));
        RpcContext.getContext().setAttachment(Cat.Context.PARENT, context.getProperty(Cat.Context.PARENT));
    }

    private Cat.Context getContext() {
        Cat.Context context = CAT_CONTEXT.get();
        if (context == null) {
            context = initContext();
            CAT_CONTEXT.set(context);
        }
        return context;
    }

    public void clearContext() {
        CAT_CONTEXT.remove();
    }

    private Cat.Context initContext() {
        Cat.Context context = new DubboCatContext();
        Map<String, String> attachments = RpcContext.getContext().getAttachments();
        if (attachments != null && attachments.size() > 0) {
            for (Map.Entry<String, String> entry : attachments.entrySet()) {
                if (Cat.Context.CHILD.equals(entry.getKey()) || Cat.Context.ROOT.equals(entry.getKey()) || Cat.Context.PARENT.equals(entry.getKey())) {
                    context.addProperty(entry.getKey(), entry.getValue());
                }
            }
        }
        return context;
    }

    private void createConsumerCross(URL url, Transaction transaction) {
        Event crossAppEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_APP, getProviderAppName(url));
        Event crossServerEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_SERVER, url.getHost());
        Event crossPortEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_PORT, url.getPort() + "");
        crossAppEvent.setStatus(Event.SUCCESS);
        crossServerEvent.setStatus(Event.SUCCESS);
        crossPortEvent.setStatus(Event.SUCCESS);
        completeEvent(crossAppEvent);
        completeEvent(crossPortEvent);
        completeEvent(crossServerEvent);
        transaction.addChild(crossAppEvent);
        transaction.addChild(crossPortEvent);
        transaction.addChild(crossServerEvent);
    }

    private void createProviderCross(Transaction transaction) {
        String consumerAppName = RpcContext.getContext().getAttachment(CommonConstants.APPLICATION_KEY);
        if (StringUtils.isEmpty(consumerAppName)) {
            consumerAppName = RpcContext.getContext().getRemoteHost() + ":" + RpcContext.getContext().getRemotePort();
        }
        Event crossAppEvent = Cat.newEvent(CatConstants.PROVIDER_CALL_APP, consumerAppName);
        Event crossServerEvent = Cat.newEvent(CatConstants.PROVIDER_CALL_SERVER, RpcContext.getContext().getRemoteHost());
        crossAppEvent.setStatus(Event.SUCCESS);
        crossServerEvent.setStatus(Event.SUCCESS);
        completeEvent(crossAppEvent);
        completeEvent(crossServerEvent);
        transaction.addChild(crossAppEvent);
        transaction.addChild(crossServerEvent);
    }

    private void completeEvent(Event event) {
        AbstractMessage message = (AbstractMessage) event;
        message.setCompleted(true);
    }
}
