package cn.com.duibaboot.ext.autoconfigure.rocketmq.sleuth;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.PostConstruct;

import brave.ErrorParser;
import brave.Span;
import brave.Tracer;
import brave.Tracing;
import brave.propagation.TraceContext;
import brave.propagation.TraceContextOrSamplingFlags;
import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.common.message.MessageExt;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;

/**
 * sleuth对rocketmq的支持
 */
@Aspect
public class SleuthRocketMqPlugin {


	@Autowired
	private Tracer tracer;
	@Autowired
	private ErrorParser errorParser;
	@Autowired
	private Tracing tracing;

	private TraceContext.Injector<RocketMqRequestTextMap> injector;
	private TraceContext.Extractor<RocketMqRequestTextMap> extractor;

	@PostConstruct
	public void init() {
		this.injector = tracing.propagation()
				.injector((carrier, key, value) -> {
					carrier.put(key, value);
				});
		this.extractor = tracing.propagation()
				.extractor((carrier, key) -> carrier.get(key));
	}

	@Around("execution(* org.apache.rocketmq.client.producer.DefaultMQProducer.send(..))")
	public Object rocketMqSendJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable{
		Span curSpan = tracer.currentSpan();
		if(curSpan == null || curSpan.isNoop()){
			return joinPoint.proceed();
		}

		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		String methodName = signature.getMethod().getName();
		Object[] args = joinPoint.getArgs();
		if(args == null || args.length < 1 || args[0] == null || !(args[0] instanceof Message)) {
			return joinPoint.proceed();
		}

		Span span = tracer.nextSpan().name("rocketmq-send")
				.kind(Span.Kind.PRODUCER)
				.remoteServiceName("rocketmq")
				.start();
		try(Tracer.SpanInScope scope = tracer.withSpanInScope(span)) {
			Message msg = (Message)args[0];
			injector.inject(span.context(), new RocketMqRequestTextMap(msg));
			if(!span.isNoop()) {
				span.tag("lc", "rocketmqProducer");//本地组件名
				span.tag("rocketmq.topic", msg.getTopic());//topic
				span.tag("rocketmq.tags", org.apache.commons.lang3.StringUtils.defaultString(msg.getTags()));//tags
				span.tag("rocketmq.type", "send");
				span.tag("producer.isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
				span.tag("producer.thread", Thread.currentThread().getName());
				span.tag("producer.serviceGroupKey", org.apache.commons.lang3.StringUtils.defaultString(ServiceGroupContext.getCurrentGroupKey()));
			}
			return joinPoint.proceed();
		} catch(Exception e){
			errorParser.error(e, span);
			throw e;
		} finally {
			span.finish();
		}
	}

	@Around("execution(* org.apache.rocketmq.client.consumer.listener.MessageListener*.consumeMessage(..))")
	public Object rocketMqConsumeJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable{
		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		String methodName = signature.getMethod().getName();
		Object[] args = joinPoint.getArgs();
		if(args == null || args.length < 1 || args[0] == null || !(args[0] instanceof List)) {
			return joinPoint.proceed();
		}
		List<?> list = (List<?>)args[0];
		if(list == null || list.isEmpty() || !(list.get(0) instanceof MessageExt)) {
			return joinPoint.proceed();
		}

		MessageExt msg = (MessageExt)list.get(0);
		Span span = null;
		if(list.size() == 1) {
			TraceContextOrSamplingFlags extracted = extractor.extract(new RocketMqRequestTextMap(msg));
			span = tracer.nextSpan(extracted).name("rocketmq-consume")
					.kind(Span.Kind.SERVER)
					.remoteServiceName("rocketmq")//远程服务名
					.start();
		}
		if(span == null) {
			span = tracer.nextSpan().name("rocketmq-consume")
					.kind(Span.Kind.SERVER)
					.remoteServiceName("rocketmq")//远程服务名
					.start();
		}

		try(Tracer.SpanInScope scope = tracer.withSpanInScope(span)) {
			if(!span.isNoop()) {
				span.tag("lc", "rocketmqConsumer");//本地组件名
//            span.tag("peer.host", "www.duiba.com.cn:3306");//远程host
				span.tag("rocketmq.topic", msg.getTopic());//topic
				span.tag("rocketmq.tags", org.apache.commons.lang3.StringUtils.defaultString(msg.getTags()));//tags
				span.tag("rocketmq.type", "consume");
				span.tag("consumer.thread", Thread.currentThread().getName());
				//TODO 可能由于aop顺序不对的原因，consumer.isPerfTest、consumer.serviceGroupKey有可能打印出错误的值
				span.tag("consumer.isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
				span.tag("consumer.serviceGroupKey", org.apache.commons.lang3.StringUtils.defaultString(ServiceGroupContext.getCurrentGroupKey()));
			}
			return joinPoint.proceed();
		} catch(Exception e){
			errorParser.error(e, span);
			throw e;
		} finally {
			span.finish();
		}
	}

	class RocketMqRequestTextMap {

		private final Message delegate;

		RocketMqRequestTextMap(Message delegate) {
			this.delegate = delegate;
		}

		public Iterator<Map.Entry<String, String>> iterator() {
			return Optional.ofNullable(delegate.getProperties()).orElse(Collections.emptyMap())
					.entrySet().iterator();
		}

		public String get(String key) {
			return delegate.getUserProperty(key);
		}

		public void put(String key, String value) {
			if (!StringUtils.hasText(value)) {
				return;
			}
			delegate.putUserProperty(key, value);
		}

	}

}
