package cn.com.duibaboot.ext.autoconfigure.rocketmq;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.rocketmq.grouping.RocketMqMessageFilter;
import cn.com.duibaboot.ext.autoconfigure.rocketmq.grouping.RocketMqMessageListenerPostProcessor4Group;
import cn.com.duibaboot.ext.autoconfigure.rocketmq.grouping.RocketMqProducerServiceGroupAspect;
import cn.com.duibaboot.ext.autoconfigure.rocketmq.perftest.RocketMqPerfAspect;
import com.alibaba.ttl.TransmittableThreadLocal;
import com.netflix.discovery.EurekaClient;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.MessageListener;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import javax.servlet.DispatcherType;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;

/**
 * RocketMQ生产者/消费者自动配置
 *
 * @author wenqi.huang
 * @date 2017/04/18
 */
@Configuration
@ConditionalOnClass({DefaultMQProducer.class})
@EnableConfigurationProperties(RocketMqProperties.class)
public class RocketMqAutoConfiguration {

    protected abstract static class RocketMqClientConfiguration{

        @Autowired
        protected RocketMqProperties rocketMqProperties;

    }

    /**
     * 配置Filter，在处理请求之前先标识服务分组标记
     */
    @Configuration
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    static class RocketMqMessageGroupConfiguration {

        @Bean
        public RocketMqMessageFilter rocketMqMessageGroupFilter() {
            return new RocketMqMessageFilter();
        }

        @Bean
        public FilterRegistrationBean<RocketMqMessageFilter> rocketMqMessageGroupConfigurer(RocketMqMessageFilter rocketMqMessageGroupFilter) {
            FilterRegistrationBean<RocketMqMessageFilter> registrationBean = new FilterRegistrationBean<>();
            registrationBean.setFilter(rocketMqMessageGroupFilter);
            List<String> urlPatterns = new ArrayList<>();
            urlPatterns.add(RocketMqMessageFilter.ROCKETMQ_MSG_DISPATCH_PATH);//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 4);
            return registrationBean;
        }

    }


    /**
     * 拦截spring.data.mongodb操作设置在测试环境下使用影子collection
     * <br/>
     * 这个类在没有引入perftest包的时候也应该生效，这样没有接入压测的消息消费者才能拒绝执行压测消息
     */
    @Configuration
    @ConditionalOnClass({TransmittableThreadLocal.class, DefaultMQProducer.class, Aspect.class})
    public static class RocketMqPerfConfiguration{
        @Bean
        public RocketMqPerfAspect getRocketMqPerfAspect(){
            return new RocketMqPerfAspect();
        }
    }

    /**
     * 加入AOP，多场景测试支持rocketmq, 如果当前服务器的场景ID刚好和消息中的场景ID相同，则直接处理即可；
     * 如果当前服务器的场景ID（或者当前服务器没有场景ID）和消息中的场景ID不同，则去eureka中找到具有相同场景ID的服务，并转发给对应服务处理。
     */
    @Configuration
    @ConditionalOnClass({DefaultMQProducer.class, EurekaClient.class})
    @ConditionalOnProperty(value = "eureka.client.enabled", havingValue = "true", matchIfMissing = true)
    static class RocketMqServiceGroupConfiguration{

        @Bean
        public RocketMqProducerServiceGroupAspect rocketMqProducerServiceGroupAspect(){
            return new RocketMqProducerServiceGroupAspect();
        }

        @Bean
        public static SpecifiedBeanPostProcessor<MessageListener> rocketMqMessageListenerPostProcessor4Group(){
            return new RocketMqMessageListenerPostProcessor4Group();
        }
    }

    @Configuration
    @ConditionalOnClass({DefaultMQProducer.class})
    @ConditionalOnMissingBean(name={"rocketMqProducer"})
    @ConditionalOnProperty(name="duiba.rocketmq.producer.enable", havingValue = "true", matchIfMissing = false)
    protected static class RocketMqProducerConfigurator extends RocketMqClientConfiguration{

        @Bean(name="rocketMqProducer", destroyMethod = "shutdown")
        public DefaultMQProducer rocketMqProducer() throws MQClientException {
            /**
             * 一个应用创建一个Producer，由应用来维护此对象，可以设置为全局对象或者单例<br>
             */
            DefaultMQProducerWrapper p = new DefaultMQProducerWrapper();

            /**
             * producerGroup 这个概念发送普通的消息时，作用不大，但是发送分布式事务消息时，比较关键，
             * 因为服务器会回查这个Group下的任意一个Producer
             * 建议把producerGroup设置成应用名
             */
            p.setProducerGroup(rocketMqProperties.getProducer().getGroup());
            p.setNamesrvAddr(rocketMqProperties.getNameSrvAddr());
            //p.setInstanceName("Producer");
            p.setSendMsgTimeout(rocketMqProperties.getProducer().getSendMsgTimeoutMillis());//单位：ms
            /**
             * Producer对象在使用之前必须要调用start初始化，初始化一次即可<br>
             * 注意：切记不可以在每次发送消息时，都调用start方法
             */
            p.start();

            return p;
        }
    }

    @Configuration
    @ConditionalOnClass({DefaultMQPushConsumer.class})
    protected static class RocketMqConsumerConfigurator extends RocketMqClientConfiguration{

        /**
         * 对MessageListener进行包装，增加cat监控和DBTimeProfile监控
         * @return
         */
        @Bean
        public static SpecifiedBeanPostProcessor<MessageListener> rocketmqMessageListenerPostProcessor(){
            return new RocketmqMessageListenerPostProcessor();
        }

        @Bean(name="rocketMqConsumer")
        @ConditionalOnMissingBean(name={"rocketMqConsumer"})
        @ConditionalOnProperty(name="duiba.rocketmq.consumer.enable", havingValue = "true", matchIfMissing = false)
        public DefaultMQPushConsumerWrapper rocketMqConsumer() {
            return new DefaultMQPushConsumerWrapper(rocketMqProperties, -1);
        }

        @Bean(name="extraRocketMqConsumer0")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer0"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[0].enable", havingValue = "true", matchIfMissing = false)
        public DefaultMQPushConsumerWrapper extraRocketMqConsumer0() {
            return new DefaultMQPushConsumerWrapper(rocketMqProperties, 0);
        }

        @Bean(name="extraRocketMqConsumer1")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer1"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[1].enable", havingValue = "true", matchIfMissing = false)
        public DefaultMQPushConsumerWrapper extraRocketMqConsumer1() {
            return new DefaultMQPushConsumerWrapper(rocketMqProperties, 1);
        }

        @Bean(name="extraRocketMqConsumer2")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer2"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[2].enable", havingValue = "true", matchIfMissing = false)
        public DefaultMQPushConsumerWrapper extraRocketMqConsumer2() {
            return new DefaultMQPushConsumerWrapper(rocketMqProperties, 2);
        }

        @EventListener(MainContextRefreshedEvent.class)
        @Order(Ordered.LOWEST_PRECEDENCE)//用户应用可能要加载缓存
        public void onEvent(MainContextRefreshedEvent event){
            Map<String, DefaultMQPushConsumerWrapper> map = event.getApplicationContext().getBeansOfType(DefaultMQPushConsumerWrapper.class);
            if(!map.isEmpty()){
                for(DefaultMQPushConsumerWrapper consumer : map.values()){
                    try {
                        //Consumer对象在使用之前必须要调用start初始化，初始化一次即可
                        if(consumer != null) {
                            consumer.startRun();
                        }
                    } catch (MQClientException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }

    }
}
