package cn.com.duibaboot.ext.autoconfigure.rocketmq.grouping;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.EurekaClientUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.DuibaFeignProperties;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupUtils;
import cn.com.duibaboot.ext.autoconfigure.rocketmq.MessageListenerConcurrentlyWrapper;
import cn.com.duibaboot.ext.autoconfigure.rocketmq.MessageListenerOrderlyWrapper;
import cn.com.duibaboot.ext.autoconfigure.rocketmq.RocketmqMessageListenerPostProcessor;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.discovery.EurekaClient;
import com.netflix.discovery.shared.Application;
import lombok.Builder;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.rocketmq.client.consumer.listener.*;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.common.message.MessageQueue;
import org.apache.rocketmq.common.protocol.heartbeat.MessageModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanNotOfRequiredTypeException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import javax.annotation.Resource;
import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.RejectedExecutionException;
import java.util.stream.Stream;

import static cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration.DUIBA_SERVICE_GROUP_KEY;

/**
 * 加入AOP，多场景测试支持rocketmq, 如果当前服务器的场景ID刚好和消息中的场景ID相同，则直接处理即可；
 * 如果当前服务器的场景ID（或者当前服务器没有场景ID）和消息中的场景ID不同，则去eureka中找到具有相同场景ID的服务，并转发给对应服务处理。
 */
public class RocketMqMessageListenerPostProcessor4Group implements SpecifiedBeanPostProcessor<MessageListener>, ApplicationContextAware {

    private static final Logger logger = LoggerFactory.getLogger(RocketMqMessageListenerPostProcessor4Group.class);
    private ApplicationContext applicationContext;

    @Value("${" + DUIBA_SERVICE_GROUP_KEY + ":}")
    private String currentServerGroupKey;//当前服务器的serviceGroupKey
    @Value("${spring.application.name}")
    private String currentApplicationName;//当前应用名

    @Override
    public Class<MessageListener> getBeanType() {
        return MessageListener.class;
    }

    @Override
    public Object postProcessBeforeInitialization(MessageListener bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(MessageListener bean, String beanName) throws BeansException {
        //暂时只在非线上环境生效
        if(Arrays.stream(applicationContext.getEnvironment().getActiveProfiles()).anyMatch(s -> s.startsWith("prod"))) {
            return bean;
        }

        ProxyFactory factory = new ProxyFactory();
        factory.setTarget(bean);
        factory.addAdvice(new MessageListenerMethodInterceptor(currentServerGroupKey, applicationContext, currentApplicationName));
        return factory.getProxy();
    }

    @Override
    public int getOrder() {
        return RocketmqMessageListenerPostProcessor.ORDER + 1;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    private static class MessageListenerMethodInterceptor implements MethodInterceptor {

        private String currentServerGroupKey;
        private ApplicationContext applicationContext;
        private String currentApplicationName;
        private volatile CloseableHttpClient httpClient;
        private volatile EurekaClient eurekaClient;
        private volatile long lastRefreshInstancesTime = -1;

        /**
         * 这里一定要注入applicationContext，后续再从applicationContext获取EurekaClient,如果在构造函数中就获取会有问题（过早初始化导致BeanPostProcessor来不及调用,形成强循环依赖）。
         * @param currentServerGroupKey
         * @param applicationContext
         */
        MessageListenerMethodInterceptor(String currentServerGroupKey, ApplicationContext applicationContext, String currentApplicationName){
            this.currentServerGroupKey = currentServerGroupKey;
            this.applicationContext = applicationContext;
            this.currentApplicationName = currentApplicationName;
        }

        private EurekaClient getEurekaClient(){
            if(eurekaClient == null){
                eurekaClient = applicationContext.getBean("eurekaClient", EurekaClient.class);
            }
            return eurekaClient;
        }

        private CloseableHttpClient getHttpClient(){
            if(httpClient == null){
                httpClient = applicationContext.getBean("apacheHttpClient", CloseableHttpClient.class);
            }
            return httpClient;
        }

        private Stream<InstanceInfo> getAllThisServerUpInstances() {
            long now = System.currentTimeMillis();
            if (now - lastRefreshInstancesTime > 2000) { //超过两秒没有更新eureka服务器列表，则更新一次
                //获取application之前强制eureka从服务器重新获取一次
                EurekaClientUtils.refreshRegistry(getEurekaClient());
                lastRefreshInstancesTime = now;
            }
            Application application = getEurekaClient().getApplication(currentApplicationName);
            if (application == null) {
                return Collections.<InstanceInfo>emptyList().stream();
            }
            List<InstanceInfo> instances = application.getInstancesAsIsFromEureka();
            return instances.stream().filter(instanceInfo -> instanceInfo.getStatus() == InstanceInfo.InstanceStatus.UP);
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            String methodName = invocation.getMethod().getName();
            if (methodName.startsWith("consumeMessage")
                    && invocation.getMethod().getParameterTypes().length >= 1
                    && invocation.getMethod().getParameterTypes()[0].equals(List.class)
            ){
                String beanName = resolveBeanName(invocation.getThis());
                if(beanName == null){
                    return invocation.proceed();
                }

                //广播消息不需要转发
                if (invocation.getThis() instanceof MessageListenerConcurrentlyWrapper
                        && ((MessageListenerConcurrentlyWrapper)invocation.getThis()).getConsumerProperties().getMessageModelEnum() == MessageModel.BROADCASTING) {
                    return invocation.proceed();
                } else if (invocation.getThis() instanceof MessageListenerOrderlyWrapper
                        && ((MessageListenerOrderlyWrapper)invocation.getThis()).getConsumerProperties().getMessageModelEnum() == MessageModel.BROADCASTING) {
                    return invocation.proceed();
                }

                //有任意多场景测试实例时才进入如下逻辑
                Object[] args = invocation.getArguments();
                List<MessageExt> list = (List<MessageExt>)args[0];
                Object messageContext = args[1];
                if(list != null && list.size() == 1) {
                    MessageExt m = list.get(0);
                    return consumeOne(invocation, m, beanName, messageContext);
                } else if(list != null && list.size() > 1 && containsServiceGroupMsg(list)){
                    logger.warn("【RocketMQ多场景测试无法支持一次消费多条rocketmq消息】检测到当前正在进行多场景测试，rocketmq消费者一次消费了多条消息且消息中有多场景测试消息（如需支持多场景消费请设置DefaultMQPushConsumer.setConsumeMessageBatchMaxSize 为1）");
                }
            }

            return invocation.proceed();
        }

        private String resolveBeanName(Object obj){
            if (obj instanceof MessageListenerConcurrentlyWrapper) {
                return ((MessageListenerConcurrentlyWrapper)obj).getBeanName();
            } else if (obj instanceof MessageListenerOrderlyWrapper) {
                return ((MessageListenerOrderlyWrapper)obj).getBeanName();
            } else {
                logger.warn("【RocketMQ多场景测试】无法获取beanName，无法支持rocketmq多场景测试，应该是BeanPostProcessor顺序乱了导致的");
                return null;
            }
        }

        private boolean canProcessByCurrentMachine(String serviceGroupKeyOfMessage){
            if(serviceGroupKeyOfMessage.equals(currentServerGroupKey)){//消息携带多场景测试的场景ID且当前服务器具有相同场景ID，当前服务器直接处理即可
                return true;
            }
            if(serviceGroupKeyOfMessage.startsWith(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX)){//key中携带的ip跟当前服务器ip相同，则当前服务器也直接处理
                String ipFromMessage = serviceGroupKeyOfMessage.substring(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX.length());
                if(ipFromMessage.equals(NetUtils.getLocalIp())){
                    return true;
                }
            }

            return false;
        }

        private Object consumeOne(MethodInvocation invocation, MessageExt m, String beanName, Object messageContext) throws Throwable{//NOSONAR
            String serviceGroupKeyOfMessage = m.getUserProperty(ServiceGroupUtils.DUIBA_SERVICE_GROUP_KEY);
            if (StringUtils.isNotBlank(serviceGroupKeyOfMessage)) {
                if(canProcessByCurrentMachine(serviceGroupKeyOfMessage)){//消息携带多场景测试的场景ID且当前服务器具有相同场景ID，当前服务器直接处理即可
                    logger.info("【RocketMQ多场景测试】当前收到消息的topic:{}, 多场景key:{}，与本机匹配，将由当前服务器直接处理.", m.getTopic(), serviceGroupKeyOfMessage);
                    ServiceGroupContext.setGroupKey(serviceGroupKeyOfMessage);
                    try {
                        return invocation.proceed();
                    }finally{
                        ServiceGroupContext.removeGroupKey();
                    }
                } else {
                    //消息携带多场景测试的场景ID,但当前服务器具有没有相同场景ID（或场景ID不同），则尝试寻找有相同场景ID的服务器进行转交处理; 如果找不到则转交给不带场景ID的服务器处理, 如果不带场景ID的服务器也找不到，则当前服务器直接处理
                    InstanceInfo selectedInstance = getAllThisServerUpInstances()
                            .filter(instanceInfo -> {
                                //先尝试寻找带相同场景ID的member
                                String memberServiceGroupKey = instanceInfo.getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_SERVICE_GROUP_KEY);
                                if(serviceGroupKeyOfMessage.equals(memberServiceGroupKey)){
                                    return true;
                                }
                                if(serviceGroupKeyOfMessage.startsWith(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX)){//key中携带的ip跟指定服务器ip相同，则交给指定服务器处理
                                    String ipFromMessage = serviceGroupKeyOfMessage.substring(ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX.length());
                                    String nodeIp = instanceInfo.getIPAddr();
                                    if(ipFromMessage.equals(nodeIp)){
                                        return true;
                                    }
                                }
                                return false;
                            })
                            .findAny().orElseGet(() -> {
                                //如果找不到带相同场景ID的，则找一个不带场景ID的服务器。
                                if(StringUtils.isBlank(currentServerGroupKey)){ //当前服务器不带场景ID，则返回null，让当前服务器处理
                                    return null;
                                }
                                return getAllThisServerUpInstances()
                                        .filter(instanceInfo -> {
                                            //寻找不带场景ID的服务器
                                            String memberServiceGroupKey = instanceInfo.getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_SERVICE_GROUP_KEY);
                                            return StringUtils.isBlank(memberServiceGroupKey);
                                        })
                                        .findAny().orElse(null);//如果还是找不到，则返回null，让当前服务器处理
                            });

                    if(selectedInstance != null){
                        String memberServiceGroupKey = selectedInstance.getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_SERVICE_GROUP_KEY);
                        if(serviceGroupKeyOfMessage.equals(memberServiceGroupKey)){
                            logger.info("【RocketMQ多场景测试】当前收到消息的topic:{}, 多场景key:{}，与本机不匹配，转发给具有相同场景ID的目标机器:{}进行处理.", m.getTopic(), serviceGroupKeyOfMessage, selectedInstance.getIPAddr());
                        }else {
                            logger.info("【RocketMQ多场景测试】当前收到消息的topic:{}, 多场景key:{}，没有找到对应的多场景服务实例，将转发给不带场景ID服务器：{}进行处理.", m.getTopic(), serviceGroupKeyOfMessage, selectedInstance.getIPAddr());
                        }

                        return submitConsumeTask2otherNode(m, messageContext, beanName, selectedInstance);
                    } else {//selectedMember为null则当前服务器直接处理掉
                        logger.info("【RocketMQ多场景测试】当前收到消息的topic:{}, 多场景key:{}，没有找到对应的多场景服务实例，当前服务器将会直接处理.", m.getTopic(), serviceGroupKeyOfMessage);
                        ServiceGroupContext.setGroupKey(serviceGroupKeyOfMessage);
                        try {
                            return invocation.proceed();
                        } finally {
                            ServiceGroupContext.removeGroupKey();
                        }
                    }
                }
            } else {//消息没有携带场景ID
                if (StringUtils.isBlank(currentServerGroupKey)) {//消息没有携带场景ID，且当前服务器刚好也没有携带场景ID，则当前服务器直接处理掉
                    logger.info("【RocketMQ多场景测试】当前收到消息的topic:{}, 多场景key:{}, 将由本机直接处理", m.getTopic(), serviceGroupKeyOfMessage);
                    return invocation.proceed();
                } else {//消息没有携带场景ID，但当前服务器携带了场景ID，则优先寻找没有场景ID的服务器进行转交处理，如果找不到没有场景ID的服务器或hazelcast处理失败，则当前服务器处理
                    InstanceInfo selectedInstance = getAllThisServerUpInstances()
                            .filter(instanceInfo -> StringUtils.isBlank(instanceInfo.getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_SERVICE_GROUP_KEY)))
                            .findAny().orElse(null);

                    if (selectedInstance != null){
                        logger.info("【RocketMQ多场景测试】当前收到消息的topic:{}, 多场景key:{}, 将转发给同样没有多场景ID的服务器{}进行处理", m.getTopic(), serviceGroupKeyOfMessage, selectedInstance.getIPAddr());
                        return submitConsumeTask2otherNode(m, messageContext, beanName, selectedInstance);
                    } else {//selectedMember为null则当前服务器直接处理掉
                        logger.info("【RocketMQ多场景测试】当前收到消息的topic:{}, 多场景key:{}, 没有找到对应的没有多场景Id的服务器，将由本机直接处理", m.getTopic(), serviceGroupKeyOfMessage);
                        return invocation.proceed();
                    }
                }
            }
        }

        /**
         * 把消费消息的任务提交到其他服务器中去执行。
         *
         * @param m
         * @param messageContext
         * @param beanName
         * @param selectedInstance
         * @return
         * @throws ExecutionException
         * @throws RejectedExecutionException
         * @throws InterruptedException
         */
        private Object submitConsumeTask2otherNode(MessageExt m, Object messageContext, String beanName, InstanceInfo selectedInstance)
                throws IOException {
            String rocketMQMsgDispatchUrl = selectedInstance.getHomePageUrl();
            if(StringUtils.isBlank(rocketMQMsgDispatchUrl)){//如果没有homePageUrl，则降级为自己拼装（某些python、nodejs等应用没有这个字段）
                rocketMQMsgDispatchUrl = "http://" + selectedInstance.getIPAddr() + ":" + selectedInstance.getPort();
            }
            if(!rocketMQMsgDispatchUrl.endsWith("/")){
                rocketMQMsgDispatchUrl = rocketMQMsgDispatchUrl.substring(0, rocketMQMsgDispatchUrl.length() - 1);
            }
            rocketMQMsgDispatchUrl = rocketMQMsgDispatchUrl + RocketMqMessageFilter.ROCKETMQ_MSG_DISPATCH_PATH;

            RocketMqRunInOtherNodeTask.RocketMqRunInOtherNodeTaskBuilder builder = RocketMqRunInOtherNodeTask.builder()
                    .beanName(beanName)
                    .msgs(Arrays.asList(m));
            if(messageContext instanceof ConsumeConcurrentlyContext) {
                ConsumeConcurrentlyContext c = (ConsumeConcurrentlyContext)messageContext;
                builder.messageQueue(c.getMessageQueue());
                builder.ackIndex(c.getAckIndex());
                builder.delayLevelWhenNextConsume(c.getDelayLevelWhenNextConsume());
            }else if(messageContext instanceof ConsumeOrderlyContext){
                ConsumeOrderlyContext c = (ConsumeOrderlyContext)messageContext;
                builder.messageQueue(c.getMessageQueue());
                builder.suspendCurrentQueueTimeMillis(c.getSuspendCurrentQueueTimeMillis());
                builder.autoCommit(c.isAutoCommit());
            }
            RocketMqRunInOtherNodeTask task = builder.build();

            HttpPost req = new HttpPost(rocketMQMsgDispatchUrl);
            byte[] bs = DuibaFeignProperties.DuibaFeignSerialization.HESSIAN2.serialize(task);
            ByteArrayEntity entity = new ByteArrayEntity(bs);
            req.setEntity(entity);
            try (CloseableHttpResponse resp = getHttpClient().execute(req)) {
                byte[] retBs = IOUtils.toByteArray(resp.getEntity().getContent());
                Object objectToReturn = DuibaFeignProperties.DuibaFeignSerialization.HESSIAN2.deserialize(retBs);
                return objectToReturn;
            }
        }

        /**
         * 检测是否有携带serviceGroupId的消息
         * @param list
         * @return
         */
        private boolean containsServiceGroupMsg(List<MessageExt> list){
            return list.stream().filter(messageExt -> StringUtils.isNotBlank(messageExt.getUserProperty(ServiceGroupUtils.DUIBA_SERVICE_GROUP_KEY)))
                    .count() > 0;
        }

    }

    @Builder
    public static class RocketMqRunInOtherNodeTask implements Callable<Object>, Serializable {

        @Resource
        private transient ApplicationContext applicationContext;

        private String beanName;

        private List<MessageExt> msgs;

        //ConsumeConcurrentlyContext的属性
        private MessageQueue messageQueue;
        private int delayLevelWhenNextConsume;
        private int ackIndex;

        //ConsumeOrderlyContext的属性
        private boolean autoCommit = true;
        private long suspendCurrentQueueTimeMillis = -1;

        @Override
        public Object call() {
            MessageListener listener;
            try {
                listener = applicationContext.getBean(beanName, MessageListener.class);
            } catch(NoSuchBeanDefinitionException | BeanNotOfRequiredTypeException e){
                throw new IllegalStateException("【RocketMQ多场景测试】找不到beanName为"+beanName+"的messageListener，无法消费多场景的rocketmq消息；多场景测试时请至少保证项目各个分支有一样名字的MessageListener", e);
            }

            if (listener instanceof MessageListenerConcurrently) {
                ConsumeConcurrentlyContext consumeConcurrentlyContext = new ConsumeConcurrentlyContext(messageQueue);
                consumeConcurrentlyContext.setAckIndex(ackIndex);
                consumeConcurrentlyContext.setDelayLevelWhenNextConsume(delayLevelWhenNextConsume);
                return ((MessageListenerConcurrently)listener).consumeMessage(msgs, consumeConcurrentlyContext);
            } else if (listener instanceof MessageListenerOrderly) {
                ConsumeOrderlyContext consumeOrderlyContext = new ConsumeOrderlyContext(messageQueue);
                consumeOrderlyContext.setAutoCommit(autoCommit);
                consumeOrderlyContext.setSuspendCurrentQueueTimeMillis(suspendCurrentQueueTimeMillis);
                return ((MessageListenerOrderly)listener).consumeMessage(msgs, consumeOrderlyContext);
            }else{
                throw new UnsupportedOperationException("【RocketMQ多场景测试】暂未支持的MessageListener类型：" + listener.getClass().getName());
            }
        }

        public void setApplicationContext(ApplicationContext applicationContext){
            this.applicationContext = applicationContext;
        }
    }
}
