package cn.com.duibaboot.ext.autoconfigure.rocketmq.cat;

import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Transaction;
import com.google.common.collect.Maps;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.spring.annotation.RocketMQMessageListener;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.core.annotation.AnnotationUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @author liuyao
 */
@Aspect
public class RocketmqCatAspect {

    @Around(value = "execution(* org.apache.rocketmq.client.producer.MQProducer.send*(org.apache.rocketmq.common.message.Message,..)) && args(msg,..)",argNames= "point,msg")
    public Object send(ProceedingJoinPoint point, Message msg) throws Throwable {
        return CatUtils.executeInCatTransaction(point::proceed, "RocketMqProducer", "send."+msg.getTopic());
    }

    @Around(value = "execution(* org.apache.rocketmq.client.producer.MQProducer.request(org.apache.rocketmq.common.message.Message,..)) && args(msg,..)",argNames= "point,msg")
    public Object sendMessages(ProceedingJoinPoint point, Message msg) throws Throwable {
        return CatUtils.executeInCatTransaction(point::proceed, "RocketMqProducer", "request."+msg.getTopic());
    }

    @Around(value = "execution(* org.apache.rocketmq.client.producer.MQProducer.send(java.util.Collection<org.apache.rocketmq.common.message.Message>,..)) && args(msgs,..)",argNames= "point,msgs")
    public Object sendMessages(ProceedingJoinPoint point, Collection<Message> msgs) throws Throwable {
        return CatUtils.executeInCatTransaction(point::proceed, "RocketMqProducer", "batchSendMessages");
    }

    @Around(value = "execution(* org.apache.rocketmq.spring.core.RocketMQListener.onMessage(..))")
    public Object rocketMqListener(ProceedingJoinPoint point) throws Throwable {
        RocketMQMessageListener listener = AnnotationUtils.findAnnotation(point.getTarget().getClass(),RocketMQMessageListener.class);
        String topic = Optional.ofNullable(listener).map(RocketMQMessageListener::topic).orElse("unknown-topic");
        String group = Optional.ofNullable(listener).map(RocketMQMessageListener::consumerGroup).orElse("unknown-group");
        return CatUtils.executeInCatTransaction(point::proceed, "RocketMqListener", group+"."+topic);
    }

    @Around(value = "execution(* org.apache.rocketmq.spring.core.RocketMQReplyListener.onMessage(..))")
    public Object rocketMqReplyListener(ProceedingJoinPoint point) throws Throwable {
        RocketMQMessageListener listener = AnnotationUtils.findAnnotation(point.getTarget().getClass(),RocketMQMessageListener.class);
        String topic = Optional.ofNullable(listener).map(RocketMQMessageListener::topic).orElse("unknown-topic");
        String group = Optional.ofNullable(listener).map(RocketMQMessageListener::consumerGroup).orElse("unknown-group");
        return CatUtils.executeInCatTransaction(point::proceed, "RocketMQReplyListener", group+"."+topic);
    }

    @Pointcut("execution(* org.apache.rocketmq.client.consumer.listener.MessageListenerOrderly.consumeMessage(java.util.List,..))")
    public void messageListenerOrderly(){}

    @Pointcut("execution(* org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently.consumeMessage(java.util.List,..))")
    public void messageListenerConcurrently(){}

    @Around(value = "(messageListenerOrderly() || messageListenerConcurrently()) && args(msgs,..)",argNames = "point,msgs")
    public Object consumeMessage(ProceedingJoinPoint point,List<MessageExt> msgs) throws Throwable {
        Map<String, Transaction> transactions = Maps.newHashMap();
        for(MessageExt message:msgs){
            if(transactions.containsKey(message.getTopic())){
                continue;
            }
            transactions.put(message.getTopic(), Cat.newTransaction("MessageQueue", "consume."+message.getTopic()));
        }
        try {
            Object result = point.proceed();
            for(Map.Entry<String,Transaction> entry:transactions.entrySet()){
                entry.getValue().setStatus(com.dianping.cat.message.Message.SUCCESS);
            }
            return result;
        } catch (Throwable e) {
            Cat.logError(e);
            for(Map.Entry<String,Transaction> entry:transactions.entrySet()){
                entry.getValue().setStatus(e);
            }
            throw e;
        }finally{
            for(Map.Entry<String,Transaction> entry:transactions.entrySet()){
                entry.getValue().complete();
            }
        }
    }

}
