package cn.com.duibaboot.ext.stream.resolver;

import org.springframework.core.MethodParameter;
import org.springframework.core.ResolvableType;
import org.springframework.lang.Nullable;
import org.springframework.messaging.Message;
import org.springframework.messaging.converter.MessageConversionException;
import org.springframework.messaging.converter.MessageConverter;
import org.springframework.messaging.converter.SmartMessageConverter;
import org.springframework.messaging.handler.annotation.support.MessageMethodArgumentResolver;
import org.springframework.messaging.handler.annotation.support.MethodArgumentTypeMismatchException;
import org.springframework.messaging.support.ErrorMessage;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.Type;

public class SmartMessageMethodArgumentResolver extends MessageMethodArgumentResolver {

    private final MessageConverter messageConverter;

    public SmartMessageMethodArgumentResolver() {
        this(null);
    }

    /**
     * Create a resolver instance with the given {@link MessageConverter}.
     * @param converter the MessageConverter to use (may be {@code null})
     * @since 4.3
     */
    public SmartMessageMethodArgumentResolver(@Nullable MessageConverter converter) {
        this.messageConverter = converter;
    }

    @Override
    public Object resolveArgument(MethodParameter parameter, Message<?> message)
            throws Exception {
        Class<?> targetMessageType = parameter.getParameterType();
        Class<?> targetPayloadType = getPayloadType(parameter);

        if (!targetMessageType.isAssignableFrom(message.getClass())) {
            throw new MethodArgumentTypeMismatchException(message, parameter,
                    "Actual message type '" + ClassUtils.getDescriptiveType(message)
                            + "' does not match expected type '"
                            + ClassUtils.getQualifiedName(targetMessageType) + "'");
        }

        Class<?> payloadClass = message.getPayload().getClass();

        if (message instanceof ErrorMessage
                || conversionNotRequired(payloadClass, targetPayloadType)) {
            return message;
        }
        Object payload = message.getPayload();
        if (isEmptyPayload(payload)) {
            throw new MessageConversionException(message,
                    "Cannot convert from actual payload type '"
                            + ClassUtils.getDescriptiveType(payload)
                            + "' to expected payload type '"
                            + ClassUtils.getQualifiedName(targetPayloadType)
                            + "' when payload is empty");
        }

        payload = convertPayload(message, parameter, targetPayloadType);
        return MessageBuilder.createMessage(payload, message.getHeaders());
    }

    private boolean conversionNotRequired(Class<?> a, Class<?> b) {
        return b == Object.class ? ClassUtils.isAssignable(a, b)
                : ClassUtils.isAssignable(b, a);
    }

    private Class<?> getPayloadType(MethodParameter parameter) {
        Type genericParamType = parameter.getGenericParameterType();
        ResolvableType resolvableType = ResolvableType.forType(genericParamType)
                .as(Message.class);
        return resolvableType.getGeneric().toClass();
    }

    @Override
    protected boolean isEmptyPayload(@Nullable Object payload) {
        if (payload == null) {
            return true;
        }
        else if (payload instanceof byte[]) {
            return ((byte[]) payload).length == 0;
        }
        else if (payload instanceof String) {
            return !StringUtils.hasText((String) payload);
        }
        else {
            return false;
        }
    }

    private Object convertPayload(Message<?> message, MethodParameter parameter,
                                  Class<?> targetPayloadType) {
        Object result = null;
        if (this.messageConverter instanceof SmartMessageConverter) {
            SmartMessageConverter smartConverter = (SmartMessageConverter) this.messageConverter;
            result = smartConverter.fromMessage(message, targetPayloadType, parameter);
        }
        else if (this.messageConverter != null) {
            result = this.messageConverter.fromMessage(message, targetPayloadType);
        }

        if (result == null) {
            throw new MessageConversionException(message,
                    "No converter found from actual payload type '"
                            + ClassUtils.getDescriptiveType(message.getPayload())
                            + "' to expected payload type '"
                            + ClassUtils.getQualifiedName(targetPayloadType) + "'");
        }
        return result;
    }

}