package cn.com.duiba.cloud.duiba.activity.service.api.dto.task;

import cn.com.duiba.cloud.biz.tool.utils.EnumUtil;
import cn.com.duiba.cloud.duiba.activity.service.api.dto.BaseDto;
import cn.com.duiba.cloud.duiba.activity.service.api.enums.task.TaskCycleUnitEnum;
import cn.com.duiba.cloud.duiba.activity.service.api.enums.task.TaskStatusEnum;
import cn.com.duiba.cloud.duiba.activity.service.api.enums.task.TaskSwitchEnum;
import cn.com.duiba.cloud.duiba.activity.service.api.enums.task.TaskTypeEnum;
import lombok.Getter;
import lombok.Setter;
import org.jetbrains.annotations.Nullable;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 任务表
 *
 * @TableName tb_task
 */
@Setter
@Getter
public class TaskDto extends BaseDto {

    private static final long serialVersionUID = 1L;

    /**
     * 归属应用
     */
    private Long appId;

    /**
     * 任务类型，1-跳转链接，2-接口任务，3-签到任务，4-分享任务
     *
     * @see TaskTypeEnum
     */
    private Integer taskType;

    /**
     * 任务状态，0-停用，1-启用
     *
     * @see TaskStatusEnum
     */
    private Integer taskStatus;

    /**
     * 任务标题
     */
    private String taskTitle;

    /**
     * 任务位开关，0位-是否有奖励，1-是否手动领奖，2位-是否全部完成后再发奖
     *
     * @see TaskSwitchEnum
     */
    private Integer taskSwitch;

    /**
     * 任务文字性描述，如副标题、图标、描述、按钮文案
     */
    private TaskDescConfig descConfig;

    /**
     * 任务有效期开启时间
     */
    private Date validStart;

    /**
     * 任务有效期结束时间
     */
    private Date validEnd;

    /**
     * 任务周期配置，如周期计算、完成次数限制、每日不可参与时间段配置
     */
    private TaskCycleConfig taskCycleConfig;

    /**
     * 任务详情，如跳转链接、分享链接、
     */
    private TaskDetail taskDetail;

    /**
     * 外部关联id，如活动id
     */
    private Long relationId;


    /**
     * 奖励配置列表
     */
    private List<TaskRewardConfigDto> rewardConfigDtoList = new ArrayList<>();


    /**
     * 任务标识
     */
    private String taskCode;

    /**
     * 返回周期序号,从0开始
     *
     * @param taskStartTime 任务开始时间
     * @param recordTime    用户参与/完成时间
     *
     * @return 周期序号, 从0开始
     */
    public static Integer calculateCycleNo(TaskCycleConfig taskCycleConfig, Date taskStartTime, Date recordTime) {
        final Boolean cycle = taskCycleConfig.getCycle();
        final Integer taskCycleUnit = taskCycleConfig.getTaskCycleUnit();
        final Integer taskCycleDays = taskCycleConfig.getTaskCycleDays();

        return calculateCycleNo(taskStartTime, recordTime, cycle, taskCycleUnit, taskCycleDays);
    }

    /**
     * 返回周期序号,从0开始
     *
     * @param taskStartTime 任务开始时间
     * @param recordTime    任务完成时间
     * @param cycle         是否循环周期
     * @param taskCycleUnit 自然周期
     * @param taskCycleDays 固定周期
     *
     * @return
     */
    @Nullable
    private static Integer calculateCycleNo(Date taskStartTime,
                                            Date recordTime,
                                            Boolean cycle,
                                            Integer taskCycleUnit,
                                            Integer taskCycleDays) {
        LocalDate startDate = taskStartTime.toInstant()
                .atZone(ZoneId.systemDefault())
                .toLocalDate();
        LocalDate endDate = recordTime.toInstant()
                .atZone(ZoneId.systemDefault())
                .toLocalDate();

        Integer cycleNo;
        if (!Objects.equals(cycle, Boolean.TRUE)) {
            cycleNo = 0;
        } else {
            if (taskCycleUnit != null) {
                // 自然周期
                TaskCycleUnitEnum taskCycleUnitEnum = EnumUtil.getByCode(TaskCycleUnitEnum.class, taskCycleUnit);
                cycleNo = Optional.ofNullable(TaskCycleUnitEnum.calculateNaturalPeriodBetween(taskCycleUnitEnum, startDate, endDate)).map(Long::intValue).orElse(null);
            } else {
                // 固定周期
                int until = (int) ChronoUnit.DAYS.between(
                        startDate,
                        endDate);
                cycleNo = until == 0 ? 0 : (until - 1) / taskCycleDays;
            }
        }

        return cycleNo;
    }

}