package cn.com.duiba.cloud.duiba.activity.service.api.remoteservice.playways;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.duiba.activity.service.api.remoteservice.playways.dto.PairDTO;
import cn.com.duiba.cloud.duiba.activity.service.api.remoteservice.playways.dto.PlaywayQuerytableDTO;
import cn.com.duiba.cloud.duiba.activity.service.api.remoteservice.playways.remote.RemoteQueryTableService;
import cn.com.duiba.cloud.duiba.activity.service.api.utils.Field;
import cn.com.duiba.cloud.duiba.activity.service.api.utils.PageList;
import cn.com.duiba.cloud.duiba.activity.service.api.utils.QueryTableItem;
import com.google.common.collect.HashBiMap;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.EnumUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static cn.com.duiba.cloud.duiba.activity.service.api.utils.Field.PresetField.DATA;
import static cn.com.duiba.cloud.duiba.activity.service.api.utils.Field.PresetField.EXT1;
import static cn.com.duiba.cloud.duiba.activity.service.api.utils.Field.PresetField.EXT2;
import static cn.com.duiba.cloud.duiba.activity.service.api.utils.Field.PresetField.EXT3;
import static cn.com.duiba.cloud.duiba.activity.service.api.utils.Field.PresetField.EXT4;

/**
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: QueryTableImpl.java , v 0.1 2019-11-17 12:50 ZhouFeng Exp $
 */
public class QueryTableImpl<T extends Field> implements QueryTable<T> {

    private String group;

    private Class<T> fieldClass;

    private HashBiMap<T, Field.PresetField> mapper = HashBiMap.create();

    private RemoteQueryTableService queryTableService;

    private Long activityId;

    public QueryTableImpl(String group, Class<T> fieldClass, Long activityId, RemoteQueryTableService queryTableService) {
        this.group = group;
        this.fieldClass = fieldClass;
        this.activityId = activityId;
        this.queryTableService = queryTableService;
        init();
    }

    @SuppressWarnings("unchecked")
    private void init() {
        //建立双向映射

        List<Enum> enumList = EnumUtils.getEnumList((Class<Enum>) (Class) fieldClass);
        for (Enum aEnum : enumList) {
            mapper.put((T) aEnum, ((Field) aEnum).getField());
        }
    }

    @Override
    public QueryTableItem<T> newItem() {
        return new InnerItem<>(new PlaywayQuerytableDTO());
    }

    class InnerItem<T extends Field> implements QueryTableItem<T> {

        private PlaywayQuerytableDTO entity;

        public InnerItem(PlaywayQuerytableDTO entity) {
            this.entity = entity;
            entity.setGroupId(group);
        }

        @Override
        public Object get(T field) {
            Field.PresetField presetField = field.getField();
            switch (presetField) {
                case ID:
                    return entity.getId();
                case SCORE:
                    return entity.getScore();
                case DATA:
                    return entity.getStringData();
                case EXT1:
                    return entity.getExt1();
                case EXT2:
                    return entity.getExt2();
                case EXT3:
                    return entity.getExt3();
                case EXT4:
                    return entity.getExt4();
                case GMT_CREATE:
                    return entity.getGmtCreate();
                case GMT_MODIFIED:
                    return entity.getGmtModified();
            }
            return null;
        }

        @Override
        public void set(T field, Object value) {
            Field.PresetField presetField = field.getField();
            switch (presetField) {
                case SCORE:
                    //当入参是Integer时，如果直接强转成long会报错，将其转成字符串再转换成Long可以避免类型转换的问题
                    entity.setScore(Long.valueOf(String.valueOf(value)));
                    break;
                case DATA:
                    entity.setStringData(String.valueOf(value));
                    break;
                case EXT1:
                    entity.setExt1(String.valueOf(value));
                    break;
                case EXT2:
                    entity.setExt2(String.valueOf(value));
                    break;
                case EXT3:
                    entity.setExt3(String.valueOf(value));
                    break;
                case EXT4:
                    entity.setExt4(String.valueOf(value));
                    break;
                case GMT_CREATE:
                    entity.setGmtCreate((Date) value);
                    break;
                case GMT_MODIFIED:
                    entity.setGmtModified((Date) value);
                    break;
            }
        }
    }


    @Override
    public void add(QueryTableItem<T> item) {
        PlaywayQuerytableDTO entity = ((InnerItem) item).entity;
        entity.setActivityId(activityId);

        queryTableService.add(entity);
    }

    @Override
    public boolean update(long id, Field type, Object value) {

        InnerItem<Field> item = new InnerItem<>(new PlaywayQuerytableDTO());
        item.set(type, value);

        return queryTableService.updateById(activityId, group, id, item.entity);
    }

    @Override
    public boolean update(long id, QueryTableItem<T> item) {
        PlaywayQuerytableDTO entity = ((InnerItem) item).entity;

        return queryTableService.updateById(activityId, group, id, entity);
    }

    @Override
    public boolean delete(long id) {
        return queryTableService.deleteById(activityId, group, id);
    }

    @Override
    public boolean increase(long id, long delta, long max) {
        return queryTableService.increase(activityId, group, id, delta, max);
    }

    @Override
    public List<QueryTableItem<T>> query(Field type1, Object value1, Field type2, Object value2) {
        List<PairDTO<Integer, Object>> queryList = new ArrayList<>(2);
        if(null != type1){
            queryList.add(PairDTO.from(type1.getField().getId(), value1));
        }
        if(null != type2){
            queryList.add(PairDTO.from(type2.getField().getId(), value2));
        }
        if(CollectionUtils.isEmpty(queryList)){
            return Collections.emptyList();
        }

        List<PlaywayQuerytableDTO> result = queryTableService.querySorted(activityId, group, queryList, 8, 1, 1, 2000);
        return toItemList(result);
    }

    @Override
    public List<QueryTableItem<T>> queryList(Field type, List<Object> values) throws BizException {
        if (CollectionUtils.isEmpty(values)) {
            return Collections.emptyList();
        } else if (values.size() > 200) {
            throw new BizException("批量查询长度不得大于200！");
        }
        return toItemList(queryTableService.batchQuery(activityId, group, type.getField().getId(), values));
    }

    @Override
    public PageList<QueryTableItem<T>> pageQuery(Field queryType, Object value, long pageNum, long pageSize) {
        return pageQuery(queryType, value, null, null, pageNum, pageSize);
    }

    @Override
    public PageList<QueryTableItem<T>> inexplicitPageQuery(Field queryType, String value, long pageNum, long pageSize) {
        List<PairDTO<Integer, Object>> queryList = Collections.emptyList();
        if (queryType != null) {
            queryList = Collections.singletonList(PairDTO.from(queryType.getField().getId(), value));
        }

        long totalCount = queryTableService.inexplicitCount(activityId, group, queryList);
        if (totalCount == 0) {
            return PageList.of(0, pageSize, Collections.emptyList());
        }

        List<PlaywayQuerytableDTO> result = queryTableService.inexplicitQuerySorted(activityId, group, queryList,
                Field.PresetField.ID.getId(), 1, pageNum, pageSize);
        return PageList.of(totalCount, pageSize, toItemList(result));
    }

    @Override
    public List<QueryTableItem<T>> queryNoSort(Field type1, Object value1, Field type2, Object value2) {
        List<PairDTO<Integer, Object>> queryList = new ArrayList<>(2);
        queryList.add(PairDTO.from(type1.getField().getId(), value1));
        queryList.add(PairDTO.from(type2.getField().getId(), value2));
        List<PlaywayQuerytableDTO> result = queryTableService.query(activityId, group, queryList, 1, 2000);
        return toItemList(result);
    }

    @Override
    public PageList<QueryTableItem<T>> pageQueryNoSort(Field queryType, Object value, long pageNum, long pageSize) {
        List<PairDTO<Integer, Object>> queryList = Collections.emptyList();
        if (queryType != null) {
            queryList = Collections.singletonList(PairDTO.from(queryType.getField().getId(), value));
        }
        long totalCount = queryTableService.count(activityId, group, queryList);
        if (totalCount == 0) {
            return PageList.of(0, pageSize, Collections.emptyList());
        }
        List<PlaywayQuerytableDTO> result = queryTableService.query(activityId, group, queryList, pageNum, pageSize);
        return PageList.of(totalCount, pageSize, toItemList(result));
    }

    @Override
    public PageList<QueryTableItem<T>> inexplicitPageQueryNoSort(Field queryType, String value, long pageNum, long pageSize) {
        List<PairDTO<Integer, Object>> queryList = Collections.emptyList();
        if (queryType != null) {
            queryList = Collections.singletonList(PairDTO.from(queryType.getField().getId(), value));
        }
        long totalCount = queryTableService.inexplicitCount(activityId, group, queryList);
        if (totalCount == 0) {
            return PageList.of(0, pageSize, Collections.emptyList());
        }
        List<PlaywayQuerytableDTO> result = queryTableService.inexplicitQuery(activityId, group, queryList, pageNum, pageSize);
        return PageList.of(totalCount, pageSize, toItemList(result));
    }

    @Override
    public PageList<QueryTableItem<T>> pageQuery(Field type1, Object value1, Field type2, Object value2, long pageNum, long pageSize) {
        if (type1 == null && type2 == null) {
            return PageList.of(0, pageSize, Collections.emptyList());
        }
        List<PairDTO<Integer, Object>> queryList = new ArrayList<>(2);
        if (type1 != null) {
            queryList.add(PairDTO.from(type1.getField().getId(), value1));
        }
        if (type2 != null) {
            queryList.add(PairDTO.from(type2.getField().getId(), value2));
        }

        long totalCount = queryTableService.count(activityId, group, queryList);
        if (totalCount == 0) {
            return PageList.of(0, pageSize, Collections.emptyList());
        }

        List<PlaywayQuerytableDTO> result = queryTableService.querySorted(activityId, group, queryList, Field.PresetField.GMT_CREATE.getId(), 1, pageNum, pageSize);

        return PageList.of(totalCount, pageSize, toItemList(result));
    }

    @Override
    public PageList<QueryTableItem<T>> pageQueryNoExt(Field queryType, Object value, Field sortField, SortType sortType, long pageNum, long pageSize)
            throws BizException {
        Field.PresetField presetField = sortField.getField();
        if (Objects.equals(EXT1, presetField)
                || Objects.equals(EXT2, presetField)
                || Objects.equals(EXT3, presetField)
                || Objects.equals(EXT4, presetField)) {
            throw new BizException("不支持 EXT 字段排序");
        }
        if (Objects.equals(DATA, presetField)) {
            throw new BizException("不支持 DATA 字段排序");
        }
        List<PairDTO<Integer, Object>> queryList = Collections.emptyList();
        if (queryType != null) {
            queryList = Collections.singletonList(PairDTO.from(queryType.getField().getId(), value));
        }

        long totalCount = queryTableService.count(activityId, group, queryList);
        if (totalCount == 0) {
            return PageList.of(0, pageSize, Collections.emptyList());
        }

        List<PlaywayQuerytableDTO> result = queryTableService.querySorted(activityId, group, queryList, sortField.getField().getId(), sortType == SortType.DESC ? 1 : 2, pageNum, pageSize);
        return PageList.of(totalCount, pageSize, toItemList(result));
    }

    private List<QueryTableItem<T>> toItemList(List<PlaywayQuerytableDTO> entities) {
        List<QueryTableItem<T>> list = new ArrayList<>(entities.size());
        for (PlaywayQuerytableDTO entity : entities) {
            list.add(new InnerItem<>(entity));
        }
        return list;
    }
}
