package cn.com.duiba.cloud.biz.tool.exception;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.message.BaseError;
import cn.com.duiba.wolf.entity.JsonResult;
import cn.hutool.core.io.IoUtil;
import cn.hutool.http.ContentType;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.lang.NonNull;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.UndeclaredThrowableException;
import java.nio.charset.StandardCharsets;

/**
 * WebMvc全局异常处理
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @since 0.0.12
 **/
@Slf4j
@Order(Ordered.LOWEST_PRECEDENCE - 11)
public class WebMvcExceptionHandler implements HandlerExceptionResolver {
    @Override
    public ModelAndView resolveException(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response
            , Object handler, @NonNull Exception ex) {
        response.setContentType(ContentType.build(ContentType.JSON, StandardCharsets.UTF_8));
        try {
            if (ex instanceof BizException) {
                return bizExceptionHandler(ex, response);
            }
            if (ex instanceof UndeclaredThrowableException) {
                if (ex.getCause() instanceof BizException) {
                    return bizExceptionHandler(ex.getCause(), response);
                } else {
                    return globalExceptionHandler(ex, request, response);
                }
            }
            if (ex instanceof MethodArgumentNotValidException) {
                return methodArgumentNotValidExceptionHandler(ex, response);
            }
            if (ex instanceof IllegalArgumentException) {
                return illegalArgumentExceptionHandler(ex, response);
            }
            return globalExceptionHandler(ex, request, response);
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), ex);
        }
    }

    private ModelAndView bizExceptionHandler(Throwable t, HttpServletResponse response) throws IOException {
        log.warn("BizException [{}]", t.getMessage(), t);
        ioWrite(response.getOutputStream(), JSON.toJSONString(
                JsonResult.fail(((BizException) t).getCode(), t.getMessage())
                , SerializerFeature.WriteMapNullValue));
        return new ModelAndView();
    }

    private ModelAndView methodArgumentNotValidExceptionHandler(Throwable t, HttpServletResponse response)
            throws IOException {
        BindingResult bindResult = null;
        if (t instanceof BindException) {
            bindResult = ((BindException) t).getBindingResult();
        } else if (t instanceof MethodArgumentNotValidException) {
            bindResult = ((MethodArgumentNotValidException) t).getBindingResult();
        }
        String message = "参数错误";
        if (bindResult != null && bindResult.hasErrors()) {
            ObjectError objectError = bindResult.getAllErrors().get(0);
            message = objectError.getDefaultMessage();
        }
        ioWrite(response.getOutputStream(), JSON.toJSONString(
                JsonResult.fail(BaseError.PARAM_ERROR.getCode(), message)
                , SerializerFeature.WriteMapNullValue));
        return new ModelAndView();
    }

    private ModelAndView illegalArgumentExceptionHandler(Throwable t, HttpServletResponse response) throws IOException {
        log.error("IllegalArgumentException", t);
        ioWrite(response.getOutputStream(), JSON.toJSONString(JsonResult.fail(t.getMessage())
                , SerializerFeature.WriteMapNullValue));
        return new ModelAndView();
    }

    private ModelAndView globalExceptionHandler(Throwable t, HttpServletRequest request
            , HttpServletResponse response) throws IOException {
        String message = t.getMessage();
        if (t instanceof InvocationTargetException && message == null) {
            message = ((InvocationTargetException) t).getTargetException().getMessage();
        }
        log.error("Exception [{} -> {}]:", request.getRequestURI(), message, t);
        ioWrite(response.getOutputStream(), JSON.toJSONString(JsonResult.fail(BaseError.SYSTEM_ERROR.getMsg())
                , SerializerFeature.WriteMapNullValue));
        return new ModelAndView();
    }

    private void ioWrite(ServletOutputStream outputStream, String msg) {
        IoUtil.writeUtf8(outputStream, true, msg);
    }
}