package cn.com.duiba.cloud.channel.center.api.open;

import cn.com.duiba.cloud.biz.tool.utils.NumberTool;
import cn.com.duiba.cloud.channel.center.api.constant.sale.ShelfStatusEnum;
import cn.com.duiba.cloud.channel.center.api.dto.sale.AddrLimitDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.BaseSpuSaleDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SaleDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SkuSaleDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SpuExpressDTO;
import cn.com.duiba.cloud.duiba.goods.center.api.constant.MediaTypeEnum;
import cn.com.duiba.cloud.duiba.goods.center.api.constant.ReviewStatusEnum;
import cn.com.duiba.cloud.duiba.goods.center.api.constant.SpuStatusEnum;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsAttributeDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsDetailDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsImageDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.SkuDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.SpuDto;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONValidator;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 前端商品与后端商品的转换/回填
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2022/2/8 13:53
 */
public class GoodsUtil {
    public static final Logger LOGGER = LoggerFactory.getLogger(GoodsUtil.class);
    /**
     * 审核状态，默认通过
     */
    public static final Integer DEFAULT_REVIEW_STATUS = ReviewStatusEnum.PASS.getReviewStatus();

    /**********************************************param to do**********************************************/

    /**
     * 前端表单转数据库对象
     */
    public static GoodsDto convert2Goods(ItemSaveParam itemSaveParam) {
        SpuDto spuDto = convert2Spu(itemSaveParam);
        List<SkuDto> skuDtoList = convert2Sku(itemSaveParam);

        GoodsDto goodsDto = new GoodsDto();
        goodsDto.setSpuDto(spuDto);
        goodsDto.setSkuList(skuDtoList);
        return goodsDto;
    }

    private static SpuDto convert2Spu(final ItemSaveParam param) {
        SpuDto spuDto = new SpuDto();
        spuDto.setId(param.getSpuId());
        spuDto.setBrandId(NumberTool.tryParseLong(param.getBrandId(), 0L));
        spuDto.setSpuType(param.getSpuType());
        spuDto.setSpuStatus(SpuStatusEnum.VALID.getSpuStatus());

        List<String> leafCategoryIds = param.getLeafCategoryIds();
        spuDto.setLeafCategoryId(NumberTool.tryParseLong(leafCategoryIds.get(leafCategoryIds.size() - 1)));

        List<GoodsAttributeDto> attributeDtoList = convert2GoodsAttribute(param.getSpuAttributes());
        spuDto.setAttributeList(attributeDtoList);
        spuDto.setImageList(convert2Image(param));
        spuDto.setDetail(convert2Detail(param));
        return spuDto;
    }

    private static List<GoodsImageDto> convert2Image(ItemSaveParam param) {
        List<GoodsImageDto> result = new ArrayList<>();

        String smallImage = param.getSmallImage();
        if (StringUtils.isNotBlank(smallImage)) {
            GoodsImageDto goodsImageDto = new GoodsImageDto();
            goodsImageDto.setPicUrl(smallImage);
            goodsImageDto.setImgType(MediaTypeEnum.IMAGE_SMALL.getMediaType());
            goodsImageDto.setSort(1L);
            goodsImageDto.setReviewStatus(DEFAULT_REVIEW_STATUS);

            result.add(goodsImageDto);
        }

        List<String> multiImages = param.getMultiImages();
        for (int i = 0; i < multiImages.size(); i++) {
            if (StringUtils.isBlank(multiImages.get(i))) {
                continue;
            }
            GoodsImageDto imageDto = new GoodsImageDto();
            imageDto.setPicUrl(multiImages.get(i));
            imageDto.setImgType(MediaTypeEnum.IMAGE.getMediaType());
            imageDto.setSort(i + 1L);
            imageDto.setReviewStatus(DEFAULT_REVIEW_STATUS);
            result.add(imageDto);
        }

        List<String> multiVideos = param.getMultiVideos();
        for (int i = 0; i < multiVideos.size(); i++) {
            if (StringUtils.isBlank(multiVideos.get(i))) {
                continue;
            }
            GoodsImageDto imageDto = new GoodsImageDto();
            imageDto.setPicUrl(multiVideos.get(i));
            imageDto.setImgType(MediaTypeEnum.VIDEO.getMediaType());
            imageDto.setSort(i + 1L);
            imageDto.setReviewStatus(DEFAULT_REVIEW_STATUS);
            result.add(imageDto);
        }

        List<ItemSaveParam.Media> qualifiedImages = param.getQualifiedImages();
        for (int i = 0; i < qualifiedImages.size(); i++) {
            ItemSaveParam.Media media = qualifiedImages.get(i);
            if (StringUtils.isBlank(media.getUrl())) {
                continue;
            }
            GoodsImageDto imageDto = new GoodsImageDto();
            imageDto.setPicUrl(media.getUrl());
            imageDto.setImgType(MediaTypeEnum.QUALIFIED.getMediaType());
            imageDto.setSort(i + 1L);
            imageDto.setReviewStatus(DEFAULT_REVIEW_STATUS);

            JSONObject extra = new JSONObject();
            extra.put("name", media.getName());
            extra.put("type", media.getType());
            extra.put("size", media.getSize());
            imageDto.setExtraInfo(extra.toJSONString());
            result.add(imageDto);
        }

        return result;
    }

    private static List<SkuDto> convert2Sku(final ItemSaveParam param) {
        List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> attributes = param.getAttributes();

        List<SkuDto> result = new ArrayList<>();
        List<List<SkuAttributeNode<SkuAttributeNode.SkuInfo>>> lists = new AttributeResolver.SkuInfoAttributeResolver().resolveNodes(attributes);
        // sku属性排序需要跨sku
        long startIndex = 1L;
        for (List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> list : lists) {

            SkuAttributeNode<SkuAttributeNode.SkuInfo> leaf = list.get(list.size() - 1);
            SkuAttributeNode.SkuInfo skuInfo = leaf.getSkuInfo();

            SkuDto skuDto = new SkuDto();
            skuDto.setId(NumberTool.tryParseLong(skuInfo.getSkuId()));
            skuDto.setSpuId(param.getSpuId());
            skuDto.setSkuCode(skuInfo.getMerchantCoding());

            List<GoodsAttributeDto> attributeDtoList = list.stream()
                    .map(SkuAttributeNode::toDto)
                    .filter(goodsAttributeDto -> StringUtils.isNotBlank(goodsAttributeDto.getAttrName()))
                    .collect(Collectors.toList());
            for (GoodsAttributeDto goodsAttributeDto : attributeDtoList) {
                goodsAttributeDto.setSort(startIndex++);
            }
            skuDto.setAttributeList(attributeDtoList);

            list.stream()
                    .filter(node -> StringUtils.isNotBlank(node.getImgUrl()))
                    .findFirst()
                    .ifPresent(node -> {
                        GoodsImageDto goodsImageDto = new GoodsImageDto();
                        goodsImageDto.setPicUrl(node.getImgUrl());
                        goodsImageDto.setImgType(MediaTypeEnum.IMAGE.getMediaType());
                        goodsImageDto.setSort(1L);
                        goodsImageDto.setReviewStatus(DEFAULT_REVIEW_STATUS);
                        skuDto.setImageList(Lists.newArrayList(goodsImageDto));
                    });

            GoodsDetailDto goodsDetailDto = new GoodsDetailDto();
            goodsDetailDto.setSuggestMarketPrice(NumberTool.tryValueOf(skuInfo.getSuggestMarketPrice()));
            goodsDetailDto.setGeneralName(param.getTitle());
            skuDto.setDetail(goodsDetailDto);

            result.add(skuDto);
        }
        return result;
    }

    private static GoodsDetailDto convert2Detail(ItemSaveParam param) {
        GoodsDetailDto goodsDetailDto = new GoodsDetailDto();
        goodsDetailDto.setGeneralName(param.getTitle());
        goodsDetailDto.setDetail(param.getDescription());
        goodsDetailDto.setSellingPoint(param.getSellingPoint());
        goodsDetailDto.setSubtitle(param.getSubtitle());
        goodsDetailDto.setReviewStatus(DEFAULT_REVIEW_STATUS);

        return goodsDetailDto;
    }

    /**
     * 前端属性转属性bean
     */
    private static List<GoodsAttributeDto> convert2GoodsAttribute(List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> nodes) {
        List<GoodsAttributeDto> attributeDtoList = nodes.stream()
                .map(SkuAttributeNode::toDto)
                .filter(goodsAttributeDto -> StringUtils.isNotBlank(goodsAttributeDto.getAttrName()))
                .collect(Collectors.toList());
        for (int i = 0; i < attributeDtoList.size(); i++) {
            GoodsAttributeDto goodsAttributeDto = attributeDtoList.get(i);
            goodsAttributeDto.setSort(i + 1L);
        }
        return attributeDtoList;
    }

    /********************************************** do to param **********************************************/
    /**
     * 数据库对象转前端详情
     */
    public static void assembleGoodsInfo(final ItemSaveParam param, final GoodsDto goodsDto) {
        if (goodsDto == null) {
            return;
        }
        assembleSpuInfo(param, goodsDto.getSpuDto());
        assembleSkuInfo(param, goodsDto.getSkuList());
    }

    /**
     * 填充spu信息
     */
    private static void assembleSpuInfo(final ItemSaveParam param, SpuDto spuDto) {
        if (spuDto == null) {
            return;
        }
        param.setSpuId(spuDto.getId());
        param.setSpuType(spuDto.getSpuType());

        Long leafCategoryId = spuDto.getLeafCategoryId();
        param.setLeafCategoryIds(Lists.newArrayList(NumberTool.tryValueOf(leafCategoryId)));
        param.setBrandId(NumberTool.tryValueOf(spuDto.getBrandId()));

        GoodsDetailDto detail = spuDto.getDetail();
        if (detail != null) {
            param.setTitle(detail.getGeneralName());
            param.setSubtitle(detail.getSubtitle());
            param.setSellingPoint(detail.getSellingPoint());
            param.setDescription(detail.getDetail());
        }
        param.setSmallImage(spuDto.getImageByImageType(MediaTypeEnum.IMAGE_SMALL).stream().findFirst().orElse(""));
        param.setMultiImages(spuDto.getImageByImageType(MediaTypeEnum.IMAGE));
        param.setMultiVideos(spuDto.getImageByImageType(MediaTypeEnum.VIDEO));


        param.setQualifiedImages(spuDto.getImageList().stream()
                                         .filter(a -> Objects.equals(MediaTypeEnum.QUALIFIED.getMediaType(), a.getImgType()))
                                         .map(qualifiedImage -> {
                                             ItemSaveParam.Media media = new ItemSaveParam.Media();
                                             media.setUrl(qualifiedImage.getPicUrl());

                                             JSONObject object = new JSONObject();
                                             if (org.apache.commons.lang.StringUtils.isNotBlank(qualifiedImage.getExtraInfo()) && JSONValidator.from(qualifiedImage.getExtraInfo()).validate()) {
                                                 object = JSONObject.parseObject(qualifiedImage.getExtraInfo());
                                             }

                                             media.setType(object.getString("type"));
                                             media.setName(object.getString("name"));
                                             media.setSize(object.getString("size"));
                                             return media;
                                         })
                                         .collect(Collectors.toList()));

        param.setSpuAttributes(spuDto.getAttributeList().stream()
                                       .map((Function<GoodsAttributeDto, SkuAttributeNode<SkuAttributeNode.SkuInfo>>) SkuAttributeNode::new)
                                       .collect(Collectors.toList()));
    }

    /**
     * 填充sku信息
     */
    private static void assembleSkuInfo(ItemSaveParam param, List<SkuDto> skuList) {
        if (CollectionUtils.isEmpty(skuList)) {
            return;
        }
        List<List<SkuAttributeNode<SkuAttributeNode.SkuInfo>>> attributeNode = convert2AttributeNode(skuList);

        // 为了避免丢失上架渠道信息，回填一下叶子节点上的已有上架渠道信息
        if (CollectionUtils.isNotEmpty(param.getAttributes())) {
            Map<String, SkuAttributeNode.SkuInfo> attributeKeyMap = param.leafSkuInfoList().stream().collect(Collectors.toMap(SkuAttributeNode.SkuInfo::getAttributeKey, Function.identity(), (a, b) -> a));
            for (SkuAttributeNode.SkuInfo skuInfo : SkuAttributeNode.leafSkuInfoList(attributeNode)) {
                String attributeKey = skuInfo.getAttributeKey();
                Optional.ofNullable(attributeKeyMap.get(attributeKey))
                        .ifPresent(existSkuInfo -> {
                            skuInfo.setFacePrice(existSkuInfo.getFacePrice());
                            skuInfo.setSalePrice(existSkuInfo.getSalePrice());
                            skuInfo.setCostPrice(existSkuInfo.getCostPrice());
                            skuInfo.setRemaining(existSkuInfo.getRemaining());
                            skuInfo.setRemainingChangeValue(existSkuInfo.getRemainingChangeValue());
                            skuInfo.setTotalRemaining(existSkuInfo.getTotalRemaining());
                            skuInfo.setOccupyRemaining(existSkuInfo.getOccupyRemaining());
                            skuInfo.setTaxRate(existSkuInfo.getTaxRate());
                            skuInfo.setShelfStatus(existSkuInfo.getShelfStatus());
                            skuInfo.setCurShelfStatus(existSkuInfo.getCurShelfStatus());
                        });
            }
        }

        param.setAttributes(new AttributeResolver.SkuInfoAttributeResolver().reverseNodes(attributeNode));
    }

    /**
     * sku属性转为前端属性，同时设置叶子结点的属性
     */
    private static List<List<SkuAttributeNode<SkuAttributeNode.SkuInfo>>> convert2AttributeNode(List<SkuDto> skuList) {
        List<List<SkuAttributeNode<SkuAttributeNode.SkuInfo>>> list = new ArrayList<>();
        // sku顺序按照属性的顺序
        for (SkuDto skuDto : skuList) {
            List<GoodsAttributeDto> attributeList = skuDto.getAttributeList();
            // 无规格商品，需要手动建立一个空白属性
            if (CollectionUtils.isEmpty(attributeList)) {
                GoodsAttributeDto goodsAttributeDto = new GoodsAttributeDto();
                goodsAttributeDto.setGeneral(skuDto);
                goodsAttributeDto.setSort(1L);
                attributeList = Lists.newArrayList(goodsAttributeDto);
                skuDto.setAttributeList(attributeList);
            }
        }
        skuList.sort(Comparator.comparing(o -> o.getAttributeList().get(0).getSort()));

        for (SkuDto skuDto : skuList) {
            // 针对叶子结点，需要设置sku属性
            SkuAttributeNode.SkuInfo skuInfo = new SkuAttributeNode.SkuInfo();
            skuInfo.setSkuId(NumberTool.tryValueOf(skuDto.getId()));
            skuInfo.setMerchantCoding(skuDto.getSkuCode());
            skuInfo.setSuggestMarketPrice(NumberTool.tryParseLong(skuDto.getDetail().getSuggestMarketPrice()));

            List<GoodsAttributeDto> attributeList = skuDto.getAttributeList();
            // 按照sku顺序排序
            attributeList.sort(Comparator.comparing(GoodsAttributeDto::getSort));

            List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> tmp = new ArrayList<>();
            for (int i = 0, last = attributeList.size() - 1; i < attributeList.size(); i++) {
                GoodsAttributeDto goodsAttributeDto = attributeList.get(i);
                SkuAttributeNode<SkuAttributeNode.SkuInfo> node = new SkuAttributeNode<>(goodsAttributeDto);
                tmp.add(node);

                boolean isLeaf = i >= last;
                if (isLeaf) {
                    node.setSkuInfo(skuInfo);
                } else {
                    node.setChildren(new ArrayList<>());
                }

                // 设置属性的图片
                if (goodsAttributeDto.hasImage() && CollectionUtils.isNotEmpty(skuDto.getImageList())) {
                    List<GoodsImageDto> imageList = skuDto.getImageList();
                    String picUrl = imageList.get(0).getPicUrl();
                    node.setImgUrl(picUrl);
                }
            }
            list.add(tmp);
        }
        return list;
    }

    /**
     * 设置上架渠道信息
     */
    public static void assembleSaleInfo(ItemSaveParam param, SaleDTO saleDTO) {
        if (saleDTO == null) {
            return;
        }

        BaseSpuSaleDTO spuSaleDTO = saleDTO.getSpuSaleDTO();
        if (spuSaleDTO != null) {
            param.setOnSale(spuSaleDTO.getShelfStatus() == null ? null : spuSaleDTO.getShelfStatus() == 1);
            param.setCurShelfStatus(spuSaleDTO.getShelfStatus());
        }


        List<AddrLimitDTO> addrLimitDTOList = saleDTO.getAddrLimitDTOList();
        if (addrLimitDTOList != null) {
            param.setAreaLimitAreaCodes(addrLimitDTOList.stream()
                                                .map(addrLimitDTO -> addrLimitDTO.getProvinceCode() + "@" + addrLimitDTO.getCityCode())
                                                .collect(Collectors.toList()));
            param.setAreaLimitSwitch(CollectionUtils.isNotEmpty(addrLimitDTOList));
            if (CollectionUtils.isNotEmpty(addrLimitDTOList)) {
                param.setAreaLimitType(addrLimitDTOList.get(0).getLimitType());
            }
        }
        SpuExpressDTO spuExpressDTO = saleDTO.getSpuExpressDTO();
        if (spuExpressDTO != null) {
            param.setExpressAlertMessage(spuExpressDTO.getExpressAlertMessage());
            param.setExpressType(spuExpressDTO.getExpressType());
            param.setExpressPrice(spuExpressDTO.getExpressPrice());
            param.setExpressTemplateId(NumberTool.tryValueOf(spuExpressDTO.getExpressTemplateId()));
            JSONObject object = Optional.ofNullable(spuExpressDTO.getExtraInfo())
                    .filter(org.apache.commons.lang.StringUtils::isNotBlank)
                    .map(JSONObject::parseObject)
                    .orElse(new JSONObject());
            param.setSupplierRemark(Optional.ofNullable(object.getString(ItemSaveParam.SUPPLIER_REMARK)).orElse(""));
        }

        // 兼容两种情况
        List<SkuSaleDTO> skuSaleDTOList = saleDTO.getSkuSaleDTOList();
        Map<Long, SkuSaleDTO> skuIdMap = Optional.ofNullable(skuSaleDTOList).orElse(Collections.emptyList()).stream()
                .collect(Collectors.toMap(SkuSaleDTO::getSkuId, Function.identity(), (a, b) -> a));
        Map<String, SkuSaleDTO> skuIdKeyMap = Optional.ofNullable(skuSaleDTOList).orElse(Collections.emptyList()).stream()
                .collect(Collectors.toMap(SkuSaleDTO::getAttributeKey, Function.identity(), (a, b) -> a));
        for (SkuAttributeNode.SkuInfo skuInfo : param.leafSkuInfoList()) {
            SkuSaleDTO skuSaleDTO;
            if (StringUtils.isNotBlank(skuInfo.getSkuId())) {
                skuSaleDTO = skuIdMap.get(NumberTool.tryParseLong(skuInfo.getSkuId()));
            } else {
                skuSaleDTO = skuIdKeyMap.get(skuInfo.getAttributeKey());
            }

            Optional.ofNullable(skuSaleDTO).ifPresent(dto -> {
                skuInfo.setSalePrice(dto.getSupplyPrice());
                skuInfo.setFacePrice(dto.getFacePrice());
                skuInfo.setCostPrice(dto.getCostPrice());
                skuInfo.setTaxRate(dto.getTaxRate());
                skuInfo.setShelfStatus(dto.getShelfStatus());
                skuInfo.setCurShelfStatus(dto.getShelfStatus());
                // 当spu的上架状态为仓库中，则sku均为仓库中；否则就是sku自己的上架状态
                if (Objects.equals(ShelfStatusEnum.UN_SHELVE.getShelfStatus(), param.getCurShelfStatus())) {
                    skuInfo.setCurShelfStatus(ShelfStatusEnum.UN_SHELVE.getShelfStatus());
                }
            });
        }
    }

    public static SaleDTO convert2SaleInfo(ItemSaveParam itemSaveParam) {
        SaleDTO saleDTO = new SaleDTO();


        return saleDTO;
    }
}
