package cn.com.duiba.cloud.channel.center.api.open;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.Conditions;
import cn.com.duiba.cloud.biz.tool.utils.NumberTool;
import cn.com.duiba.cloud.channel.center.api.constant.sale.ShelfStatusEnum;
import cn.com.duiba.cloud.channel.center.api.constant.sale.SpuAuditStatusEnum;
import cn.com.duiba.cloud.channel.center.api.dto.sale.AddrLimitDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.BaseSpuSaleDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SaleDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SkuSaleDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SpuDescConfigDto;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SpuExpressDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.StockWarnConfigDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.AbstractGoodsDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsAttributeDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsDetailDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsImageDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.SkuDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.SpuDto;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONValidator;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 商品编辑差异化更新包装类
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2022/1/9 13:34
 */
@Setter
@Getter
public class ItemCompareResult implements Serializable {
    private static final long serialVersionUID = 1L;
    public static final String AREA_CODE_REGEX = "@";
    private Long spuId;

    /**
     * spu维度信息
     */
    private SpuCompare spuCompare = new SpuCompare();
    /**
     * sku维度信息
     */
    private List<SkuCompare> skuCompareList = new ArrayList<>();

    /**
     * 审批单状态，空表示无需审批
     */
    private Integer spuAuditStatus = null;

    private Long curAppId;
    private Long curStaffId;
    private Integer curShelfStatus;

    @Setter
    @Getter
    public static class SpuCompare implements Serializable {
        private static final long serialVersionUID = 1L;

        // 来自GoodsDto
        private SpuDto spuDto;

        // 来自SaleDTO
        private BaseSpuSaleDTO spuSaleDTO;
        private BaseSpuSaleDTO spuSaleDTO4Update;
        private List<AddrLimitDTO> addrLimitDTOList;
        private SpuExpressDTO spuExpressDTO;

        private StockWarnConfigDto stockWarnConfigDto;
        private SpuDescConfigDto spuDescConfigDto;
    }


    @Setter
    @Getter
    public static class SkuCompare implements Serializable {
        private static final long serialVersionUID = 1L;
        private Boolean newSku;
        private Long skuId;
        private String attributeKey;

        // 来自GoodsDto
        private SkuDto skuDto;

        /**
         * sku上架项变化
         * 不用来计算状态，比较的时候已经计算好了
         */
        private SkuSaleDTO skuSaleDTO;
        /**
         * sku上架项状态
         */
        private SkuSaleDTO skuSaleDTO4Update;


        // 来自StockParam
        private StockSubParam stockSubParam;
    }

    @Setter
    @Getter
    public static class StockSubParam implements Serializable {
        private static final long serialVersionUID = 1L;

        /**
         * 新增时初始库存数量
         */
        private Long initNumber;

        /**
         * 修改时变更库存数量
         */
        private Long changeNumber;

        /**
         * 成本价（分）
         */
        private Long costPrice;
    }

    public ItemCompareResult(Long curAppId, Long curStaffId) {
        this.curAppId = curAppId;
        this.curStaffId = curStaffId;
    }

    /**
     * 比较数据库中的商品数据与前端提交参数，得到self
     */
    public void compare(GoodsDto goodsDto,
                        SaleDTO saleDTO,
                        ItemSaveParam itemSaveParam) throws BizException {
        this.compare4Goods(goodsDto, itemSaveParam);
        this.compare4Sale(saleDTO, itemSaveParam);
        this.compare4Stock(itemSaveParam);
        if (goodsDto == null && saleDTO == null) {

        } else {
            this.calculateSpuAuditStatus();
        }
    }

    private void compare4Goods(GoodsDto goodsDto, ItemSaveParam itemSaveParam) throws BizException {
        this.setSpuId(itemSaveParam.getSpuId());
        // 先初始化好sku信息
        for (SkuAttributeNode.SkuInfo skuInfo : itemSaveParam.leafSkuInfoList()) {
            boolean newSku = StringUtils.isBlank(skuInfo.getSkuId());
            SkuCompare skuCompare = new SkuCompare();
            this.getSkuCompareList().add(skuCompare);

            skuCompare.setNewSku(newSku);
            skuCompare.setSkuId(NumberTool.tryParseLong(skuInfo.getSkuId()));
            skuCompare.setAttributeKey(skuInfo.getAttributeKey());
        }

        GoodsDto after = GoodsUtil.convert2Goods(itemSaveParam);
        if (goodsDto == null || DiffUtil.diff(goodsDto.getSpuDto(), after.getSpuDto())
                || DiffUtil.diffSkuDto(goodsDto.getSkuList(), after.getSkuList())) {
            // 商品只要有一处不同，就整体更新
            this.spuCompare.setSpuDto(after.getSpuDto());

            // 设置Sku
            Map<String, SkuDto> skuAttributeKeyMap = after.getSkuList().stream()
                    .collect(Collectors.toMap(skuDto -> GoodsAttributeDto.getAttributeKey4Attribute(skuDto.getAttributeList()), Function.identity()));
            for (SkuCompare skuCompare : this.skuCompareList) {
                skuCompare.setSkuDto(skuAttributeKeyMap.get(skuCompare.getAttributeKey()));
            }
        }

    }

    private void compare4Sale(SaleDTO saleDTO, ItemSaveParam itemSaveParam) throws BizException {
        if (saleDTO != null) {
            this.setCurShelfStatus(saleDTO.getSpuSaleDTO().getShelfStatus());
        }
        Map<String, SkuCompare> skuKeyMap = this.getSkuCompareList().stream().collect(Collectors.toMap(SkuCompare::getAttributeKey, Function.identity()));
        Map<Long, SkuSaleDTO> skuIdMap = saleDTO == null ? Collections.emptyMap() :
                saleDTO.getSkuSaleDTOList().stream().collect(Collectors.toMap(SkuSaleDTO::getSkuId, Function.identity()));

        boolean allUnShelf = itemSaveParam.leafSkuInfoList().stream().allMatch(
                skuInfo -> Objects.equals(skuInfo.getShelfStatus(), ShelfStatusEnum.UN_SHELVE.getShelfStatus())
        );
        Conditions.expectTrue(!allUnShelf, "至少设置一个sku为上架状态");

        for (SkuAttributeNode.SkuInfo skuInfo : itemSaveParam.leafSkuInfoList()) {
            SkuCompare skuCompare = skuKeyMap.get(skuInfo.getAttributeKey());
            Long skuId = NumberTool.tryParseLong(skuInfo.getSkuId());
            SkuSaleDTO skuSaleDTOFromParam = new SkuSaleDTO();
            skuSaleDTOFromParam.setSkuId(skuId);
            skuSaleDTOFromParam.setSpuId(this.spuId);
            skuSaleDTOFromParam.setSupplyPrice(skuInfo.getSalePrice());
            skuSaleDTOFromParam.setFacePrice(skuInfo.getFacePrice());
            skuSaleDTOFromParam.setCostPrice(skuInfo.getCostPrice());
            skuSaleDTOFromParam.setTaxRate(skuInfo.getTaxRate());
            skuSaleDTOFromParam.setShelfStatus(skuInfo.getShelfStatus());
            skuSaleDTOFromParam.setSourceAppId(this.curAppId);
            skuSaleDTOFromParam.setSkuName(itemSaveParam.getTitle());
            skuSaleDTOFromParam.setShelfStatus(skuInfo.getShelfStatus());

            if (skuId == null || !skuIdMap.containsKey(skuId)) {
                // 设置渠道信息
                skuCompare.setSkuSaleDTO(skuSaleDTOFromParam);
            } else {
                SkuSaleDTO skuSaleDTO = skuIdMap.get(skuId);
                // 如果是状态变化
                if (!Objects.equals(skuSaleDTO.getShelfStatus(), skuSaleDTOFromParam.getShelfStatus())) {
                    SkuSaleDTO skuSaleDTO4Update = new SkuSaleDTO();
                    skuSaleDTO4Update.setSkuId(skuSaleDTO.getSkuId());
                    skuSaleDTO4Update.setSpuId(skuSaleDTO.getSpuId());
                    skuSaleDTO4Update.setAppId(skuSaleDTO.getAppId());

                    skuSaleDTO4Update.setShelfStatus(skuSaleDTOFromParam.getShelfStatus());
                    skuCompare.setSkuSaleDTO4Update(skuSaleDTO4Update);
                }

                if (DiffUtil.diff(this, skuSaleDTO, skuSaleDTOFromParam)) {
                    // 设置渠道信息
                    SkuSaleDTO skuSaleDTO4Save = new SkuSaleDTO();
                    skuSaleDTO4Save.setSkuId(skuSaleDTO.getSkuId());
                    skuSaleDTO4Save.setSpuId(skuSaleDTO.getSpuId());
                    skuSaleDTO4Save.setAppId(skuSaleDTO.getAppId());

                    skuSaleDTO4Save.setSupplyPrice(skuSaleDTOFromParam.getSupplyPrice());
                    skuSaleDTO4Save.setFacePrice(skuSaleDTOFromParam.getFacePrice());
                    skuSaleDTO4Save.setCostPrice(skuSaleDTOFromParam.getCostPrice());
                    skuSaleDTO4Save.setTaxRate(skuSaleDTOFromParam.getTaxRate());
                    skuSaleDTO4Save.setSkuName(skuSaleDTOFromParam.getSkuName());
                    skuSaleDTO4Save.setShelfStatus(skuSaleDTOFromParam.getShelfStatus());
                    skuSaleDTO4Save.setSourceAppId(skuSaleDTOFromParam.getSourceAppId());
                    skuCompare.setSkuSaleDTO(skuSaleDTO4Save);
                }
            }
        }

        if (saleDTO == null || DiffUtil.diff(saleDTO.getAddrLimitDTOList(), itemSaveParam.getAreaLimitAreaCodes())) {
            this.spuCompare.setAddrLimitDTOList(itemSaveParam.getAreaLimitAreaCodes().stream()
                                                        .map(areaLimitAreaCode -> {
                                                            AddrLimitDTO addrLimitDTO = convert2AddrLimitDTO(areaLimitAreaCode);
                                                            addrLimitDTO.setLimitType(itemSaveParam.getAreaLimitType());
                                                            addrLimitDTO.setSpuId(itemSaveParam.getSpuId());
                                                            return addrLimitDTO;
                                                        }).collect(Collectors.toList()));
        }

        if (saleDTO == null || DiffUtil.diff(saleDTO.getSpuExpressDTO(), itemSaveParam)) {
            SpuExpressDTO spuExpressDTO = new SpuExpressDTO();
            spuExpressDTO.setSpuId(itemSaveParam.getSpuId());

            spuExpressDTO.setExpressAlertMessage(itemSaveParam.getExpressAlertMessage());
            spuExpressDTO.setExpressType(itemSaveParam.getExpressType());
            spuExpressDTO.setExpressPrice(Optional.ofNullable(itemSaveParam.getExpressPrice()).orElse(0L));
            spuExpressDTO.setExpressTemplateId(NumberTool.tryParseLong(itemSaveParam.getExpressTemplateId()));
            JSONObject object = new JSONObject();
            object.put(ItemSaveParam.SUPPLIER_REMARK, itemSaveParam.getSupplierRemark());
            spuExpressDTO.setExtraInfo(object.toJSONString());
            this.spuCompare.setSpuExpressDTO(spuExpressDTO);
        }

        Integer shelfStatus = itemSaveParam.getOnSale() ? ShelfStatusEnum.SHELVE.getShelfStatus() : ShelfStatusEnum.UN_SHELVE.getShelfStatus();
        BaseSpuSaleDTO spuSaleDTOFromParam = new BaseSpuSaleDTO();
        spuSaleDTOFromParam.setAppId(this.curAppId);
        spuSaleDTOFromParam.setSpuId(itemSaveParam.getSpuId());
        spuSaleDTOFromParam.setSpuName(itemSaveParam.getTitle());
        spuSaleDTOFromParam.setSpuType(itemSaveParam.getSpuType());

        List<String> leafCategoryIds = itemSaveParam.getLeafCategoryIds();
        spuSaleDTOFromParam.setCategoryId(NumberTool.tryParseLong(leafCategoryIds.get(leafCategoryIds.size() - 1)));
        LongSummaryStatistics statistics = itemSaveParam.leafSkuInfoList().stream()
                .map(skuInfo -> Optional.ofNullable(skuInfo.getSalePrice()).orElse(0L))
                .collect(Collectors.summarizingLong(Long::intValue));
        spuSaleDTOFromParam.setSupplyPriceMin(statistics.getMin());
        spuSaleDTOFromParam.setSupplyPriceMax(statistics.getMax());

        statistics = itemSaveParam.leafSkuInfoList().stream()
                .map(skuInfo -> Optional.ofNullable(skuInfo.getFacePrice()).orElse(0L))
                .collect(Collectors.summarizingLong(Long::intValue));
        spuSaleDTOFromParam.setFacePriceMin(statistics.getMin());
        spuSaleDTOFromParam.setFacePriceMax(statistics.getMax());

        statistics = itemSaveParam.leafSkuInfoList().stream()
                .map(skuInfo -> Optional.ofNullable(skuInfo.getCostPrice()).orElse(0L))
                .collect(Collectors.summarizingLong(Long::intValue));
        spuSaleDTOFromParam.setCostPriceMin(statistics.getMin());
        spuSaleDTOFromParam.setCostPriceMax(statistics.getMax());

        spuSaleDTOFromParam.setShelfStatus(shelfStatus);
        spuSaleDTOFromParam.setShelfSwitch(shelfStatus);
        spuSaleDTOFromParam.setIsUsable(shelfStatus);
        spuSaleDTOFromParam.setSourceAppId(this.curAppId);
        if (saleDTO == null) {
            this.spuCompare.setSpuSaleDTO(spuSaleDTOFromParam);
        } else {
            BaseSpuSaleDTO spuSaleDTO = saleDTO.getSpuSaleDTO();
            // 如果是状态变化
            if (!Objects.equals(spuSaleDTO.getShelfStatus(), spuSaleDTOFromParam.getShelfStatus())) {
                BaseSpuSaleDTO exist = saleDTO.getSpuSaleDTO();
                BaseSpuSaleDTO spuSaleDTO4Update = new BaseSpuSaleDTO();
                spuSaleDTO4Update.setSpuId(exist.getSpuId());
                spuSaleDTO4Update.setAppId(exist.getAppId());
                spuSaleDTO4Update.setShelfStatus(spuSaleDTOFromParam.getShelfStatus());
                spuSaleDTO4Update.setShelfSwitch(spuSaleDTOFromParam.getShelfSwitch());
                spuSaleDTO4Update.setIsUsable(spuSaleDTOFromParam.getIsUsable());
                this.spuCompare.setSpuSaleDTO4Update(spuSaleDTO4Update);
            }

            if (DiffUtil.diff(spuSaleDTO, spuSaleDTOFromParam)) {
                BaseSpuSaleDTO spuSaleDTO4Save = new BaseSpuSaleDTO();
                spuSaleDTO4Save.setAppId(spuSaleDTO4Save.getAppId());
                spuSaleDTO4Save.setSpuId(spuSaleDTO4Save.getSpuId());

                spuSaleDTO4Save.setSpuName(spuSaleDTOFromParam.getSpuName());
                spuSaleDTO4Save.setSpuType(spuSaleDTOFromParam.getSpuType());
                spuSaleDTO4Save.setCategoryId(spuSaleDTOFromParam.getCategoryId());

                spuSaleDTO4Save.setSourceAppId(spuSaleDTOFromParam.getSourceAppId());
                spuSaleDTO4Save.setLastAuditStatus(spuSaleDTOFromParam.getLastAuditStatus());

                spuSaleDTO4Save.setSupplyPriceMin(spuSaleDTOFromParam.getSupplyPriceMin());
                spuSaleDTO4Save.setSupplyPriceMax(spuSaleDTOFromParam.getSupplyPriceMax());
                spuSaleDTO4Save.setFacePriceMin(spuSaleDTOFromParam.getFacePriceMin());
                spuSaleDTO4Save.setFacePriceMax(spuSaleDTOFromParam.getFacePriceMax());
                spuSaleDTO4Save.setCostPriceMin(spuSaleDTOFromParam.getCostPriceMin());
                spuSaleDTO4Save.setCostPriceMax(spuSaleDTOFromParam.getCostPriceMax());

                spuSaleDTO4Save.setShelfStatus(spuSaleDTOFromParam.getShelfStatus());
                spuSaleDTO4Save.setShelfSwitch(spuSaleDTOFromParam.getShelfSwitch());
                spuSaleDTO4Save.setIsUsable(spuSaleDTOFromParam.getIsUsable());
                this.spuCompare.setSpuSaleDTO(spuSaleDTO4Save);
            }
        }
    }

    private static AddrLimitDTO convert2AddrLimitDTO(String areaLimitAreaCode) {
        String[] areaCodes = areaLimitAreaCode.split(AREA_CODE_REGEX);
        int length = areaCodes.length;
        AddrLimitDTO addrLimitDTO = new AddrLimitDTO();
        addrLimitDTO.setProvinceName("");
        addrLimitDTO.setProvinceCode(areaCodes[0]);

        if (length == 1) {
            addrLimitDTO.setCityName(AddrLimitDTO.ALL_CITY_NAME);
            addrLimitDTO.setCityCode(AddrLimitDTO.ALL_CITY_CODE);
        } else if (length == 2) {
            addrLimitDTO.setCityName("");
            addrLimitDTO.setCityCode(areaCodes[1]);
        }
        return addrLimitDTO;
    }

    private void compare4Stock(ItemSaveParam itemSaveParam) {
        Map<String, SkuCompare> skuKeyMap = this.getSkuCompareList().stream().collect(Collectors.toMap(SkuCompare::getAttributeKey, Function.identity()));
        for (SkuAttributeNode.SkuInfo skuInfo : itemSaveParam.leafSkuInfoList()) {
            SkuCompare skuCompare = skuKeyMap.get(skuInfo.getAttributeKey());
            StockSubParam stockSubParam = new StockSubParam();
            stockSubParam.setCostPrice(skuInfo.getCostPrice());
            if (skuCompare.newSku) {
                stockSubParam.setInitNumber(Optional.ofNullable(skuInfo.getTotalRemaining()).orElse(0L));
            } else if (skuInfo.getRemainingChangeValue() != null && skuInfo.getRemainingChangeValue() != 0) {
                stockSubParam.setChangeNumber(skuInfo.getRemainingChangeValue());
            }
            skuCompare.setStockSubParam(stockSubParam);
        }

    }

    /**
     * 带有优先级的设置
     */
    private void setSpuAuditStatus(Integer spuAuditStatus) {
        if (spuAuditStatus == null) {
            return;
        }
        if (this.spuAuditStatus == null || Objects.equals(this.spuAuditStatus, SpuAuditStatusEnum.PASS.getAuditStatus())) {
            // 只有原来为NULL，或者为自动通过，允许不为NULL的覆盖
            this.spuAuditStatus = spuAuditStatus;
        }
    }

    public GoodsDto convert2GoodsDto() {
        GoodsDto goodsDto = new GoodsDto();
        goodsDto.setSpuDto(this.spuCompare.getSpuDto());
        goodsDto.setSkuList(this.skuCompareList.stream().map(SkuCompare::getSkuDto).filter(Objects::nonNull).collect(Collectors.toList()));
        return goodsDto;

    }

    public SaleDTO convert2Sale() {
        SaleDTO saleDTO = new SaleDTO();
        saleDTO.setSpuId(this.spuId);
        saleDTO.setSpuSaleDTO(this.spuCompare.getSpuSaleDTO());

        saleDTO.setSkuSaleDTOList(this.skuCompareList.stream().map(skuCompare -> {
            SkuSaleDTO skuSaleDTO = skuCompare.getSkuSaleDTO();
            if (skuSaleDTO == null) {
                return null;
            }
            skuSaleDTO.setSkuId(skuCompare.getSkuId());
            skuSaleDTO.setAttributeKey(skuCompare.getAttributeKey());
            return skuSaleDTO;
        }).filter(Objects::nonNull).collect(Collectors.toList()));
        saleDTO.setAddrLimitDTOList(this.spuCompare.getAddrLimitDTOList());
        saleDTO.setSpuExpressDTO(this.spuCompare.getSpuExpressDTO());
        return saleDTO;
    }

    public Map<String, StockSubParam> convert2StockChange() {
        if (CollectionUtils.isEmpty(this.getSkuCompareList())) {
            return Collections.emptyMap();
        }
        return this.getSkuCompareList().stream()
                .filter(skuCompare -> skuCompare.getStockSubParam() != null)
                .collect(Collectors.toMap(SkuCompare::getAttributeKey, SkuCompare::getStockSubParam));
    }

    /**
     * 在编辑商品情况下，来判断本次状态
     */
    private void calculateSpuAuditStatus() {
        if (this.spuCompare.getSpuSaleDTO4Update() != null) {
            this.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
            return;
        }
        boolean curOnSale = Objects.equals(this.curShelfStatus, ShelfStatusEnum.SHELVE.getShelfStatus());
        if (!curOnSale) {
            // 当前状态是下架时，只有上架才需要审批
            this.setSpuAuditStatus(null);
            return;
        }

        // spu维度，这里没有判断spuSaleDTO，因为spuSaleDTO的信息都来自sku，所以不影响状态，只要更新就好了
        if (this.spuCompare.getSpuDto() != null || this.spuCompare.getSpuExpressDTO() != null
                || this.spuCompare.getAddrLimitDTOList() != null) {
            this.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
            return;
        }

        // sku维度
        for (SkuCompare skuCompare : this.skuCompareList) {
            if (skuCompare.newSku) {
                // 上架且新增sku
                this.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
                continue;
            }

            // 商品信息
            if (skuCompare.getSkuDto() != null) {
                this.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
            }

            // 上架状态
            if (skuCompare.getSkuSaleDTO4Update() != null) {
                this.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
            }

            if (skuCompare.getSkuSaleDTO() != null) {
                // 其他情况在diff的时候就设置好了 DiffUtil.diff(ItemCompareResult, SkuSaleDTO, SkuSaleDTO)
            }

            // 库存
            StockSubParam stockSubParam = skuCompare.getStockSubParam();
            if (stockSubParam != null) {
                if (stockSubParam.getChangeNumber() < 0) {
                    this.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
                } else {
                    this.setSpuAuditStatus(SpuAuditStatusEnum.PASS.getAuditStatus());
                }
            }

        }
    }


    private static class DiffUtil {

        public static boolean diff(SpuDto before, SpuDto after) throws BizException {
            if (before == null) {
                return true;
            }

            Conditions.expectTrue(Objects.equals(before.getSpuType(), after.getSpuType()), "不允许修改商品类型");

            boolean spuDiff = !Objects.equals(before.getBrandId(), after.getBrandId())
                    || !Objects.equals(before.getLeafCategoryId(), after.getLeafCategoryId())
                    || !Objects.equals(before.getSpuType(), after.getSpuType());
            if (spuDiff) {
                return true;
            }

            if (diff4AbstractGoods(before, after, false)) {
                return true;
            }
            return false;
        }

        private static boolean diff4AbstractGoods(AbstractGoodsDto before,
                                                  AbstractGoodsDto after,
                                                  boolean sku) throws BizException {
            if (diffAttribute(before.getAttributeList(), after.getAttributeList(), sku)) {
                return true;
            }
            if (diffMedia(before.getImageList(), after.getImageList())) {
                return true;
            }
            if (diffDetail(before.getDetail(), after.getDetail())) {
                return true;
            }
            return false;
        }

        private static boolean diffDetail(GoodsDetailDto before, GoodsDetailDto after) {
            GoodsDetailDto goodsDetailDto = new GoodsDetailDto();
            goodsDetailDto.setSellingPoint("");
            goodsDetailDto.setSubtitle("");
            goodsDetailDto.setDetail("");
            goodsDetailDto.setSuggestMarketPrice("");
            goodsDetailDto.setGeneralName("");
            boolean diff = !Objects.equals(before.getGeneralName(), after.getGeneralName())
                    || !Objects.equals(before.getDetail(), after.getDetail())
                    || !Objects.equals(before.getSubtitle(), after.getSubtitle())
                    || !Objects.equals(before.getSuggestMarketPrice(), after.getSuggestMarketPrice())
                    || !Objects.equals(before.getSellingPoint(), after.getSellingPoint());
            if (diff) {
                return true;
            }

            return false;
        }

        private static boolean diffMedia(List<GoodsImageDto> before, List<GoodsImageDto> after) {
            if (before == null) {
                return true;
            }
            if (before.size() != after.size()) {
                return true;
            }
            Function<GoodsImageDto, String> function =
                    dto -> dto.getImgType() + dto.getPicUrl() + dto.getSort() + Optional.ofNullable(dto.getExtraInfo()).orElse("");
            return !Objects.equals(before.stream().map(function).collect(Collectors.joining()), after.stream().map(function).collect(Collectors.joining()));
        }


        private static boolean diffAttribute(List<GoodsAttributeDto> before,
                                             List<GoodsAttributeDto> after,
                                             boolean sku) throws BizException {
            if (before == null) {
                return true;
            }

            if (sku) {
                if (before.size() != after.size()) {
                    throw new BizException("sku属性数量不允许修改");
                }
                Function<GoodsAttributeDto, String> function = GoodsAttributeDto::getAttrName;
                if (!after.stream().map(function).collect(Collectors.toList()).containsAll(before.stream().map(function).collect(Collectors.toList()))) {
                    throw new BizException("sku属性值不允许减少");
                }
            }

            if (before.size() != after.size()) {
                return true;
            }

            Function<GoodsAttributeDto, String> function = dto -> dto.getAttrName() + dto.getAttrValue();
            return !Objects.equals(before.stream().map(function).collect(Collectors.joining()), after.stream().map(function).collect(Collectors.joining()));
        }

        public static boolean diffSkuDto(List<SkuDto> before, List<SkuDto> after) throws BizException {
            if (before == null) {
                return true;
            }
            if (before.size() != after.size()) {
                return true;
            }
            if (after.stream().anyMatch(skuDto -> skuDto.getId() == null)) {
                return true;
            }

            Map<Long, SkuDto> skuDtoMap = before.stream().collect(Collectors.toMap(SkuDto::getId, Function.identity()));
            for (SkuDto skuDto : after) {
                Long skuId = skuDto.getId();
                if (skuId == null) {
                    return true;
                }
                SkuDto dto = skuDtoMap.get(skuId);
                if (dto == null) {
                    return true;
                }
                if (diff(dto, skuDto)) {
                    return true;
                }
            }

            // 顺序是否发生变化
            for (int i = 0; i < before.size(); i++) {
                if (!Objects.equals(before.get(i).getId(), after.get(i).getId())) {
                    return true;
                }
            }


            return false;
        }

        private static boolean diff(SkuDto before, SkuDto after) throws BizException {
            boolean diff = !Objects.equals(before.getSkuCode(), after.getSkuCode());
            if (diff) {
                return true;
            }
            if (diff4AbstractGoods(before, after, true)) {
                return true;
            }
            return false;
        }

        public static boolean diff(ItemCompareResult itemCompareResult,
                                   SkuSaleDTO before,
                                   SkuSaleDTO after) {
            if (before == null) {
                return true;
            }

            if (Objects.equals(itemCompareResult.getCurShelfStatus(), ShelfStatusEnum.SHELVE.getShelfStatus())) {
                // 这里多做一步，如果降价，就设置为自动通过
                if (!Objects.equals(before.getSupplyPrice(), after.getSupplyPrice())) {
                    if (before.getSupplyPrice() > after.getSupplyPrice()) {
                        itemCompareResult.setSpuAuditStatus(SpuAuditStatusEnum.PASS.getAuditStatus());
                    } else {
                        itemCompareResult.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
                    }
                }
                if (!Objects.equals(before.getFacePrice(), after.getFacePrice())
                        || !Objects.equals(before.getCostPrice(), after.getCostPrice())
                        || !Objects.equals(before.getTaxRate(), after.getTaxRate())) {
                    // 其他发生变化
                    itemCompareResult.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
                }
            }
            return !Objects.equals(before.getSupplyPrice(), after.getSupplyPrice())
                    || !Objects.equals(before.getFacePrice(), after.getFacePrice())
                    || !Objects.equals(before.getCostPrice(), after.getCostPrice())
                    || !Objects.equals(before.getTaxRate(), after.getTaxRate());
        }

        public static boolean diff(List<AddrLimitDTO> before, List<String> after) {
            if (before == null) {
                return true;
            }
            if (before.size() != after.size()) {
                return true;
            }
            Function<AddrLimitDTO, String> function = addrLimitDTO -> {
                String code = addrLimitDTO.getProvinceCode();
                if (!Objects.equals(AddrLimitDTO.ALL_CITY_CODE, addrLimitDTO.getCityCode())) {
                    code += AREA_CODE_REGEX + addrLimitDTO.getCityCode();
                }
                return code;
            };
            return !Objects.equals(before.stream().map(function).collect(Collectors.joining()),
                                   after.stream().map(ItemCompareResult::convert2AddrLimitDTO).map(function).collect(Collectors.joining()));

        }

        public static boolean diff(SpuExpressDTO before, ItemSaveParam after) {
            if (before == null) {
                return true;
            }
            String supplierRemark = "";
            if (StringUtils.isNotBlank(before.getExtraInfo()) && JSONValidator.from(before.getExtraInfo()).validate()) {
                supplierRemark = JSONObject.parseObject(before.getExtraInfo()).getString(ItemSaveParam.SUPPLIER_REMARK);
            }
            return !Objects.equals(before.getExpressAlertMessage(), after.getExpressAlertMessage())
                    || !Objects.equals(before.getExpressAlertMessage(), after.getExpressAlertMessage())
                    || !Objects.equals(Optional.ofNullable(supplierRemark).orElse(""), after.getSupplierRemark())
                    || !Objects.equals(before.getExpressType(), after.getExpressType())
                    || !Objects.equals(before.getExpressPrice(), after.getExpressPrice())
                    || !Objects.equals(before.getExpressTemplateId(),
                                       Optional.ofNullable(after.getExpressTemplateId())
                                               .map(NumberTool::tryParseLong).orElse(null));
        }

        public static boolean diff(BaseSpuSaleDTO before, BaseSpuSaleDTO after) {
            if (before == null) {
                return true;
            }
            return !Objects.equals(before.getSupplyPriceMax(), after.getSupplyPriceMax())
                    || !Objects.equals(before.getSupplyPriceMin(), after.getSupplyPriceMin())
                    || !Objects.equals(before.getFacePriceMax(), after.getFacePriceMax())
                    || !Objects.equals(before.getFacePriceMin(), after.getFacePriceMin())
                    || !Objects.equals(before.getCostPriceMax(), after.getCostPriceMax())
                    || !Objects.equals(before.getCostPriceMin(), after.getCostPriceMin())
                    ;
        }
    }
}
