package cn.com.duiba.cloud.channel.center.api.open;


import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.NumberTool;
import cn.com.duiba.cloud.channel.center.api.constant.sale.ShelfStatusEnum;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsAttributeDto;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;

import javax.validation.ConstraintViolation;
import javax.validation.Valid;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import javax.validation.constraints.Max;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * sku规格,存在父子级联关系
 * copy from：cn.com.duiba.goods.center.api.remoteservice.dto.sku.SkuAttributeNode
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/11/11 11:31 上午
 */
@Getter
@Setter
@SuppressWarnings("AlibabaRemoveCommentedCode")
public class SkuAttributeNode<T extends Serializable> implements Serializable {
    private static final long serialVersionUID = 9077699683796064327L;

    /**
     * 规格id/属性Id,如「颜色」,specId
     *
     * @deprecated 不以该Id作为标志
     */
    @Deprecated
    private String attrNameId;

    /**
     * 考虑到无规格商品
     */
    // @NotBlank(message = "属性名称不能为空")
    private String attrName;

    /**
     * 属性值id,如「红色」,
     *
     * @deprecated 不以该Id作为标志
     */
    @Deprecated
    private String attrValueId;

    /**
     * 属性值,如「红色」
     */
    // @NotBlank(message = "属性值不能为空")
    private String attrValue;

    /**
     * 图片地址
     * 目前只有根节点有
     */
    private String imgUrl;

    /**
     * 销售信息（T可能为SKU信息、SKU积分换购信息）
     * 目前只有根节点有
     */
    private T skuInfo;

    /**
     * 子属性列表
     */
    private List<SkuAttributeNode<T>> children;


    public SkuAttributeNode(String attrNameId, String attrName, String attrValueId, String attrValue) {
        this.attrNameId = attrNameId;
        this.attrName = attrName;
        this.attrValueId = attrValueId;
        this.attrValue = attrValue;
    }


    public SkuAttributeNode() {
    }

    public SkuAttributeNode(GoodsAttributeDto goodsAttributeDto) {
        this(goodsAttributeDto.getAttrId(),
             goodsAttributeDto.getAttrName(),
             goodsAttributeDto.getAttrValueId(),
             goodsAttributeDto.getAttrValue());
    }

    public SkuAttributeNode(Long attrNameId, String attrName, Long attrValueId, String attrValue) {
        this.attrNameId = NumberTool.tryValueOf(attrNameId);
        this.attrName = attrName;
        this.attrValueId = NumberTool.tryValueOf(attrValueId);
        this.attrValue = attrValue;
    }

    public GoodsAttributeDto toDto() {
        GoodsAttributeDto goodsAttributeDto = new GoodsAttributeDto();
        goodsAttributeDto.setAttrId(NumberTool.tryParseLong(getAttrNameId()));
        goodsAttributeDto.setAttrName(getAttrName());
        goodsAttributeDto.setAttrValue(getAttrValue());
        goodsAttributeDto.setAttrValueId(NumberTool.tryParseLong(getAttrValueId()));
        goodsAttributeDto.setHasImage(StringUtils.isNotBlank(getImgUrl()));
        return goodsAttributeDto;
    }

    /**
     * 销售信息
     */
    @Getter
    @Setter
    @Valid
    public static class SkuInfo implements Serializable {

        private static final long serialVersionUID = -9491789203686703L;
        /**
         * skuId，更新时使用
         */
        private String skuId;

        /**
         * 建议市场价(仅供管理后台用)
         */
        @Min(value = 0, message = "建议零售价不允许小于0")
        private Long suggestMarketPrice;

        /**
         * 展示价格，划线价
         */
        @Min(value = 0, message = "划线价不允许小于0")
        @NotNull(message = "划线价必填")
        private Long facePrice;

        /**
         * 兑换价格/供货价
         */
        @Min(value = 0, message = "供货价不允许小于0")
        @NotNull(message = "供货价必填")
        private Long salePrice;

        /**
         * 成本价
         */
        @Min(value = 0, message = "成本价不允许小于0")
        private Long costPrice;
        /**
         * 商家编码/供应商商品编码
         */
        private String merchantCoding;

        /**
         * 图片链接（暂时取的是第一个规格属性配置的imgUrl）
         */
        private String imgUrl;

        /**
         * 剩余库存，可销售库存
         */
        private Long remaining;

        /**
         * 前一次库存
         *
         * @deprecated 直接算到remainingChangeValue里吧
         */
        @Deprecated
        private Long preRemaining;

        /**
         * 库存改变值
         * 编辑的时候需要
         */
        private Long remainingChangeValue;

        /**
         * 总库存
         * 也是新建sku的库存
         */
        @Min(value = 0, message = "库存不能为负数")
        @NotNull(message = "库存必填")
        private Long totalRemaining;
        /**
         * 预占库存,>0说明有预占库存
         * 有预占的情况不允许删除该SKU
         */
        private Long occupyRemaining;

        /**
         * 税率，范围
         */
        @Max(value = 100, message = "税率不能超过100")
        @Min(value = 0, message = "税率不能为负数")
        private Short taxRate;

        /**
         * 上架状态(0:未上架,1:已上架)
         *
         * @see ShelfStatusEnum
         */
        private Integer shelfStatus;

        /**
         * 当前上架状态,新创建的sku无此值
         * 上架状态(0:未上架,1:已上架)
         */
        private Integer curShelfStatus;

        /**
         * 属性key，后端用，用于分辨sku
         */
        private String attributeKey;


        public void valid() throws BizException {
            ValidatorFactory vf = Validation.buildDefaultValidatorFactory();
            Validator validator = vf.getValidator();
            Set<ConstraintViolation<SkuInfo>> set = validator.validate(this);
            for (ConstraintViolation<SkuInfo> constraintViolation : set) {
                throw new BizException(constraintViolation.getMessage());
            }
        }


        /* ************************* 以下是卡券相关 ************************* */

        /**
         * 关联卡券
         */
        private Long cardLibraryId;

        /**
         * 新增批次
         */
        private Boolean createBatch;


        /**
         * 有效期 日期
         */
        private Integer startTime;
        private Integer endTime;

        /**
         * 卡号
         */
        private String cardNo;
        /**
         * 卡密
         */
        private String cardPassWord;


        /**
         * 卡券数量（重复卡券有）
         */
        private Long couponCount;


    }


    /**
     * 属性拼接成字符串
     * 用做sku标识
     */
    public static <R extends SkuAttributeNode.SkuInfo> String getAttributeKey4Node(List<SkuAttributeNode<R>> list) {
        return StringUtils.join(list.stream()
                                        .map(SkuAttributeNode::getAttrValue)
                                        .collect(Collectors.toList()),
                                ";");
    }

    public static <R extends SkuAttributeNode.SkuInfo> List<R> leafSkuInfoList(List<List<SkuAttributeNode<R>>> lists) {
        List<R> result = new ArrayList<>();
        for (List<SkuAttributeNode<R>> list : lists) {
            SkuAttributeNode<R> leaf = list.get(list.size() - 1);
            R skuInfo = leaf.getSkuInfo();
            skuInfo.setAttributeKey(getAttributeKey4Node(list));
            result.add(skuInfo);
        }
        return result;
    }
}
