package cn.com.duiba.cloud.channel.center.api.open;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.google.common.base.Joiner;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 属性工具类，列表转树，树转列表
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/11/16 3:37 下午
 */
public abstract class AttributeResolver<R extends SkuAttributeNode.SkuInfo> implements SkuResolver<R> {
    //规格属性分割符
    private static final String ATTRIBUTE_SEPARATOR = ":";
    //规格属性组分割符
    private static final String ATTRIBUTE_GROUP_SEPARATOR = ";";


    public static void main(String[] args) {

        SkuAttributeNode<SkuAttributeNode.SkuInfo> node1 = new SkuAttributeNode(1L, "颜色", 1L, "红色");
        SkuAttributeNode<SkuAttributeNode.SkuInfo> node2 = new SkuAttributeNode<>(1L, "颜色", 2L, "黄色");
        List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> firstNodes = new ArrayList<>();
        firstNodes.add(node1);
        firstNodes.add(node2);

        SkuAttributeNode<SkuAttributeNode.SkuInfo> node3 = new SkuAttributeNode<>(2L, "尺寸", 3L, "大");
        SkuAttributeNode<SkuAttributeNode.SkuInfo> node4 = new SkuAttributeNode<>(2L, "尺寸", 4L, "小");
        List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> secondNodes = new ArrayList<>();
        secondNodes.add(node3);
        secondNodes.add(node4);


        SkuAttributeNode<SkuAttributeNode.SkuInfo> node5 = new SkuAttributeNode<>(3L, "地区", 5L, "浙江");
        SkuAttributeNode<SkuAttributeNode.SkuInfo> node6 = new SkuAttributeNode<>(3L, "地区", 6L, "上海");
        List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> thirdNodes = new ArrayList<>();
        thirdNodes.add(node5);
        thirdNodes.add(node6);


        SkuAttributeNode<SkuAttributeNode.SkuInfo> node7 = new SkuAttributeNode<>(4L, "性别", 7L, "男");
        SkuAttributeNode<SkuAttributeNode.SkuInfo> node8 = new SkuAttributeNode<>(4L, "性别", 8L, "女");
        SkuAttributeNode<SkuAttributeNode.SkuInfo> node9 = new SkuAttributeNode<>(4L, "性别", 9L, "保密");
        List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> fourthNodes = new ArrayList<>();
        fourthNodes.add(node7);
        fourthNodes.add(node8);
        fourthNodes.add(node9);


        node1.setChildren(secondNodes);
        node2.setChildren(secondNodes);
        node3.setChildren(thirdNodes);
        node4.setChildren(thirdNodes);
        node5.setChildren(fourthNodes);
        node6.setChildren(fourthNodes);

        List<List<SkuAttributeNode<SkuAttributeNode.SkuInfo>>> resolveNodes = new SkuInfoAttributeResolver().resolveNodes(firstNodes);

        for (List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> skuAttributeNodes : resolveNodes) {
            List<String> collect = skuAttributeNodes.stream()
                    .map(node -> Joiner.on(":").join(node.getAttrName(), node.getAttrValue()))
                    .collect(Collectors.toList());

            System.out.println(Joiner.on(";").join(collect));
        }


        List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> list = new SkuInfoAttributeResolver().reverseNodes(resolveNodes);


        System.out.println(StringUtils.equals(JSON.toJSONString(firstNodes, SerializerFeature.DisableCircularReferenceDetect),
                                              JSON.toJSONString(list, SerializerFeature.DisableCircularReferenceDetect)));

    }

    /**
     * 树转列表
     * 红->大、小
     * 黄->大、小
     * 转为
     * (skuInfo)红、大
     * (skuInfo)红、小
     * (skuInfo)黄、大
     * (skuInfo)黄、小
     */
    @Override
    public List<List<SkuAttributeNode<R>>> resolveNodes(List<SkuAttributeNode<R>> nodes) {
        if (CollectionUtils.isEmpty(nodes)) {
            return Collections.emptyList();
        }

        List<List<SkuAttributeNode<R>>> listOfNode = new ArrayList<>();

        for (SkuAttributeNode<R> node : nodes) {
            List<String> skuAttributes = new ArrayList<>();
            addAllNodes(node, skuAttributes, StringUtils.EMPTY, listOfNode, new ArrayList<>());
        }

        return listOfNode;
    }

    /**
     * 将sku解析成树型结构
     * <p>
     * 列表转树
     * 红、大(skuInfo)
     * 红、小(skuInfo)
     * 黄、大(skuInfo)
     * 黄、小(skuInfo)
     * 转为
     * 红->大(skuInfo)、小(skuInfo)
     * 黄->大(skuInfo)、小(skuInfo)
     *
     * @param resolveNodes
     */
    @Override
    public List<SkuAttributeNode<R>> reverseNodes(List<List<SkuAttributeNode<R>>> resolveNodes) {
        if (CollectionUtils.isEmpty(resolveNodes)) {
            return Collections.emptyList();
        }
        List<SkuAttributeNode<R>> nodes = new ArrayList<>();
        for (List<SkuAttributeNode<R>> attributeNodes : resolveNodes) {
            reverseSingleSku(attributeNodes, nodes);
        }

        return nodes;
    }

    private void addAllNodes(SkuAttributeNode<R> node,
                             List<String> listOfNode,
                             String attributeJson,
                             List<List<SkuAttributeNode<R>>> skuAttributeNodes,
                             List<SkuAttributeNode<R>> tmp) {
        if (node != null) {
            attributeJson = StringUtils.join(attributeJson, node.getAttrName(), ATTRIBUTE_SEPARATOR, node.getAttrValue());

            tmp = new ArrayList<>(tmp);
            tmp.add(node);

            List<SkuAttributeNode<R>> children = node.getChildren();
            if (CollectionUtils.isEmpty(children)) {
                listOfNode.add(attributeJson);
                skuAttributeNodes.add(tmp);
            } else {
                attributeJson = StringUtils.join(attributeJson, ATTRIBUTE_GROUP_SEPARATOR);
                for (SkuAttributeNode<R> child : children) {
                    addAllNodes(child, listOfNode, attributeJson, skuAttributeNodes, tmp);
                }
            }
        }
    }

    private void reverseSingleSku(List<SkuAttributeNode<R>> attributeNodes, List<SkuAttributeNode<R>> allNodes) {
        //前节点
        SkuAttributeNode<R> prevNode = null;
        //头节点，用于设置图片
        SkuAttributeNode<R> headNode = null;
        for (int i = 0, last = attributeNodes.size() - 1; i < attributeNodes.size(); i++) {
            SkuAttributeNode<R> node = attributeNodes.get(i);
            if (prevNode == null) {
                headNode = prevNode = getOrCreateNode(node, allNodes);
            } else {
                prevNode = getOrCreateNode(node, prevNode.getChildren());
            }
            postNode(headNode, prevNode);
        }
    }

    abstract void postNode(SkuAttributeNode<R> headNode, SkuAttributeNode<R> node);

    /**
     * 获取或者创建节点
     *
     * @return
     */
    private SkuAttributeNode<R> getOrCreateNode(SkuAttributeNode<R> node,
                                                List<SkuAttributeNode<R>> nodes) {
        SkuAttributeNode<R> existNode = getExistNode(node.getAttrValue(), nodes);
        if (existNode == null) {
            existNode = node;
            nodes.add(existNode);
        }
        return existNode;
    }

    /**
     * 获取节点
     *
     * @return
     */
    private SkuAttributeNode<R> getExistNode(String attrValue, List<SkuAttributeNode<R>> nodes) {
        for (SkuAttributeNode<R> node : nodes) {
            if (Objects.equals(attrValue, node.getAttrValue())) {
                return node;
            }
        }
        return null;
    }

    public static class SkuInfoAttributeResolver<R extends SkuAttributeNode.SkuInfo> extends AttributeResolver<R> {


        @Override
        void postNode(SkuAttributeNode<R> headNode,
                      SkuAttributeNode<R> node) {
            //目前只需要根节点存储图片信息，从叶节点sku中获取图片
            SkuAttributeNode.SkuInfo skuInfo = node.getSkuInfo();
            if (skuInfo != null) {
                if (headNode.getImgUrl() == null) {
                    headNode.setImgUrl(skuInfo.getImgUrl());
                }
                // //sku中图片信息清空，减少数据传输
                // node.getSkuInfo().setImgUrl(null);
            }
        }
    }
}
