package cn.com.duiba.cloud.channel.center.api.open;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.Conditions;
import cn.com.duiba.cloud.biz.tool.utils.NumberTool;
import cn.com.duiba.cloud.channel.center.api.constant.sale.ShelfStatusEnum;
import cn.com.duiba.cloud.channel.center.api.constant.sale.SpuAuditStatusEnum;
import cn.com.duiba.cloud.channel.center.api.dto.sale.AddrLimitDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.BaseSpuSaleDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SkuSaleDTO;
import cn.com.duiba.cloud.channel.center.api.dto.sale.SpuExpressDTO;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.AbstractGoodsDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsAttributeDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsDetailDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.GoodsImageDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.SkuDto;
import cn.com.duiba.cloud.duiba.goods.center.api.dto.goods.SpuDto;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONValidator;
import org.apache.commons.lang.StringUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 描述：
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2022/4/7 10:08
 */
public class DiffUtil {
    public static boolean diff(SpuDto before, SpuDto after) throws BizException {
        if (before == null) {
            return true;
        }

        Conditions.expectTrue(Objects.equals(before.getSpuType(), after.getSpuType()), "不允许修改商品类型");

        boolean spuDiff = !Objects.equals(before.getBrandId(), after.getBrandId())
                || !Objects.equals(before.getLeafCategoryId(), after.getLeafCategoryId())
                || !Objects.equals(before.getSpuType(), after.getSpuType());
        if (spuDiff) {
            return true;
        }

        if (diff4AbstractGoods(before, after, false)) {
            return true;
        }
        return false;
    }

    private static boolean diff4AbstractGoods(AbstractGoodsDto before,
                                              AbstractGoodsDto after,
                                              boolean sku) throws BizException {
        if (diffAttribute(before.getAttributeList(), after.getAttributeList(), sku)) {
            return true;
        }
        if (diffMedia(before.getImageList(), after.getImageList())) {
            return true;
        }
        if (diffDetail(before.getDetail(), after.getDetail())) {
            return true;
        }
        return false;
    }

    private static boolean diffDetail(GoodsDetailDto before, GoodsDetailDto after) {
        GoodsDetailDto goodsDetailDto = new GoodsDetailDto();
        goodsDetailDto.setSellingPoint("");
        goodsDetailDto.setSubtitle("");
        goodsDetailDto.setDetail("");
        goodsDetailDto.setSuggestMarketPrice("");
        goodsDetailDto.setGeneralName("");
        boolean diff = !Objects.equals(before.getGeneralName(), after.getGeneralName())
                || !Objects.equals(before.getDetail(), after.getDetail())
                || !Objects.equals(before.getSubtitle(), after.getSubtitle())
                || !Objects.equals(before.getSuggestMarketPrice(), after.getSuggestMarketPrice())
                || !Objects.equals(before.getSellingPoint(), after.getSellingPoint());
        if (diff) {
            return true;
        }

        return false;
    }

    private static boolean diffMedia(List<GoodsImageDto> before, List<GoodsImageDto> after) {
        if (before == null) {
            return true;
        }
        if (before.size() != after.size()) {
            return true;
        }
        Function<GoodsImageDto, String> function =
                dto -> dto.getImgType() + dto.getPicUrl() + dto.getSort() + Optional.ofNullable(dto.getExtraInfo()).orElse("");
        return !Objects.equals(before.stream().map(function).collect(Collectors.joining()), after.stream().map(function).collect(Collectors.joining()));
    }


    private static boolean diffAttribute(List<GoodsAttributeDto> before,
                                         List<GoodsAttributeDto> after,
                                         boolean sku) throws BizException {
        if (before == null) {
            return true;
        }

        if (sku) {
            if (before.size() != after.size()) {
                throw new BizException("sku属性数量不允许修改");
            }
            Function<GoodsAttributeDto, String> function = GoodsAttributeDto::getAttrName;
            if (!after.stream().map(function).collect(Collectors.toList()).containsAll(before.stream().map(function).collect(Collectors.toList()))) {
                throw new BizException("sku属性值不允许减少");
            }
        }

        if (before.size() != after.size()) {
            return true;
        }

        Function<GoodsAttributeDto, String> function = dto -> dto.getAttrName() + dto.getAttrValue();
        return !Objects.equals(before.stream().map(function).collect(Collectors.joining()), after.stream().map(function).collect(Collectors.joining()));
    }

    public static boolean diffSkuDto(List<SkuDto> before, List<SkuDto> after) throws BizException {
        if (before == null) {
            return true;
        }
        if (before.size() != after.size()) {
            return true;
        }
        if (after.stream().anyMatch(skuDto -> skuDto.getId() == null)) {
            return true;
        }

        Map<Long, SkuDto> skuDtoMap = before.stream().collect(Collectors.toMap(SkuDto::getId, Function.identity()));
        for (SkuDto skuDto : after) {
            Long skuId = skuDto.getId();
            if (skuId == null) {
                return true;
            }
            SkuDto dto = skuDtoMap.get(skuId);
            if (dto == null) {
                return true;
            }
            if (diff(dto, skuDto)) {
                return true;
            }
        }

        // 顺序是否发生变化
        for (int i = 0; i < before.size(); i++) {
            if (!Objects.equals(before.get(i).getId(), after.get(i).getId())) {
                return true;
            }
        }


        return false;
    }

    private static boolean diff(SkuDto before, SkuDto after) throws BizException {
        boolean diff = !Objects.equals(before.getSkuCode(), after.getSkuCode());
        if (diff) {
            return true;
        }
        if (diff4AbstractGoods(before, after, true)) {
            return true;
        }
        return false;
    }

    public static boolean diff(ItemCompareResult itemCompareResult,
                               SkuSaleDTO before,
                               SkuSaleDTO after) {
        if (before == null) {
            return true;
        }

        if (Objects.equals(itemCompareResult.getCurShelfStatus(), ShelfStatusEnum.SHELVE.getShelfStatus())) {
            // 这里多做一步，如果降价，就设置为自动通过
            if (!Objects.equals(before.getSupplyPrice(), after.getSupplyPrice())) {
                if (before.getSupplyPrice() > after.getSupplyPrice()) {
                    itemCompareResult.setSpuAuditStatus(SpuAuditStatusEnum.PASS.getAuditStatus());
                } else {
                    itemCompareResult.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
                }
            }
            if (!Objects.equals(before.getFacePrice(), after.getFacePrice())
                    || !Objects.equals(before.getCostPrice(), after.getCostPrice())
                    || !Objects.equals(before.getTaxRate(), after.getTaxRate())) {
                // 其他发生变化
                itemCompareResult.setSpuAuditStatus(SpuAuditStatusEnum.WAIT.getAuditStatus());
            }
        }
        return !Objects.equals(before.getSupplyPrice(), after.getSupplyPrice())
                || !Objects.equals(before.getFacePrice(), after.getFacePrice())
                || !Objects.equals(before.getCostPrice(), after.getCostPrice())
                || !Objects.equals(before.getTaxRate(), after.getTaxRate());
    }

    public static boolean diff(List<AddrLimitDTO> before, List<String> after) {
        if (before == null) {
            return true;
        }
        if (before.size() != after.size()) {
            return true;
        }
        Function<AddrLimitDTO, String> function = addrLimitDTO -> {
            String code = addrLimitDTO.getProvinceCode();
            if (!Objects.equals(AddrLimitDTO.ALL_CITY_CODE, addrLimitDTO.getCityCode())) {
                code += ItemCompareResult.AREA_CODE_REGEX + addrLimitDTO.getCityCode();
            }
            return code;
        };
        return !Objects.equals(before.stream().map(function).collect(Collectors.joining()),
                               after.stream().map(ItemCompareResult::convert2AddrLimitDTO).map(function).collect(Collectors.joining()));

    }

    public static boolean diff(SpuExpressDTO before, ItemSaveParam after) {
        if (before == null) {
            return true;
        }
        String supplierRemark = "";
        if (StringUtils.isNotBlank(before.getExtraInfo()) && JSONValidator.from(before.getExtraInfo()).validate()) {
            supplierRemark = JSONObject.parseObject(before.getExtraInfo()).getString(ItemSaveParam.SUPPLIER_REMARK);
        }
        return !Objects.equals(before.getExpressAlertMessage(), after.getExpressAlertMessage())
                || !Objects.equals(before.getExpressAlertMessage(), after.getExpressAlertMessage())
                || !Objects.equals(Optional.ofNullable(supplierRemark).orElse(""), after.getSupplierRemark())
                || !Objects.equals(before.getExpressType(), after.getExpressType())
                || !Objects.equals(before.getExpressPrice(), after.getExpressPrice())
                || !Objects.equals(before.getExpressTemplateId(),
                                   Optional.ofNullable(after.getExpressTemplateId())
                                           .map(NumberTool::tryParseLong).orElse(null));
    }

    public static boolean diff(BaseSpuSaleDTO before, BaseSpuSaleDTO after) {
        if (before == null) {
            return true;
        }
        return !Objects.equals(before.getSupplyPriceMax(), after.getSupplyPriceMax())
                || !Objects.equals(before.getSupplyPriceMin(), after.getSupplyPriceMin())
                || !Objects.equals(before.getFacePriceMax(), after.getFacePriceMax())
                || !Objects.equals(before.getFacePriceMin(), after.getFacePriceMin())
                || !Objects.equals(before.getCostPriceMax(), after.getCostPriceMax())
                || !Objects.equals(before.getCostPriceMin(), after.getCostPriceMin())
                ;
    }
}
