package cn.com.duiba.cloud.channel.center.api.open;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.Conditions;
import cn.com.duiba.cloud.channel.center.api.constant.sale.AddrLimitTypeEnum;
import cn.com.duiba.cloud.channel.center.api.constant.sale.ShelfStatusEnum;
import cn.com.duiba.cloud.duiba.goods.center.api.constant.SpuTypeEnum;
import cn.com.duiba.cloud.duiba.goods.center.api.constant.SubSpuTypeEnum;
import lombok.Getter;
import lombok.Setter;
import org.hibernate.validator.constraints.Length;
import org.hibernate.validator.constraints.Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * 商品编辑/保存参数
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/11/11 11:20 上午
 */
@Getter
@Setter
public class ItemSaveParam implements Serializable {
    public static final Logger LOGGER = LoggerFactory.getLogger(ItemSaveParam.class);

    private static final long serialVersionUID = 1L;

    // 供应商备注
    public static final String SUPPLIER_REMARK = "supplierRemark";

    /**
     * 商品Id
     * 编辑的时候才有
     */
    private Long spuId;

    /*-------------------- 基本信息 --------------------*/
    /**
     * 商品类型(1:实物,2:优惠券:3:充值,4:权益)
     *
     * @see SpuTypeEnum
     */
    @NotNull(message = "商品类型不能为空")
    private Integer spuType;


    /**
     * 商品子类型
     *
     * @see SubSpuTypeEnum
     */
    private Integer subSpuType;

    /**
     * 类目Id
     */
    @NotNull(message = "商品类目不能为空")
    @Size(min = 1, max = 3)
    private List<String> leafCategoryIds;

    /**
     * 品牌id
     * 无品牌传0
     */
    private String brandId = "0";


    /**
     * 商品分组Id，可选
     */
    @Size(max = 50, message = "每个商品最多配置到50个分组")
    private List<String> itemClassifyIds = new ArrayList<>();


    /*-------------------- 商品信息 --------------------*/

    /**
     * spuName
     */
    @NotBlank(message = "商品名称不能为空")
    @Length(max = 100, message = "商品名称过长")
    private String title;

    /**
     * 子标题、商品子名称
     *
     * @deprecated 暂时没用到
     */
    @Deprecated
    private String subtitle;

    /**
     * 卖点描述
     */
    @Length(max = 25, message = "卖点描述过长")
    private String sellingPoint;

    /**
     * 缩略图
     */
    private String smallImage;

    /**
     * 主图
     */
    @Size(min = 1, max = 9, message = "商品主图必传一张，最多支持9张")
    private List<String> multiImages = new ArrayList<>();

    /**
     * 主图视频
     */
    private List<String> multiVideos = new ArrayList<>();


    /**
     * 供应商名称、编码
     */
    private String supplierName;
    private String supplierCode;

    /**
     * 商品编码
     */
    private String spuCode;


    /*-------------------- 商品信息 --------------------*/

    /**
     * 规则/sku
     * 顺序就是展示顺序
     */
    @Size(min = 1)
    @Valid
    private List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> attributes = new ArrayList<>();


    /*-------------------- 商品参数 --------------------*/


    /**
     * 商品参数
     */
    @Size(max = 50, message = "参数项最多选择50个")
    @Valid
    private List<SkuAttributeNode<SkuAttributeNode.SkuInfo>> spuAttributes = new ArrayList<>();

    /**
     * 图文详情，支持富文本
     */
    @Length(max = 30000, message = "图文详情过长")
    private String description;

    /**
     * 商品资质
     */
    @Size(max = 5, message = "商品资质最多5张")
    private List<Media> qualifiedImages = new ArrayList<>();


    /*-------------------- 上下架设置 --------------------*/

    /**
     * 上架设置 = shelfStatus
     * true-立即上架 false-放入仓库
     */
    @NotNull(message = "上架设置不允许为空")
    private Boolean onSale = false;

    /**
     * 当前上架状态
     *
     * @see ShelfStatusEnum
     */
    private Integer curShelfStatus;


    /**
     * 是否开启订单审核
     */
    private Boolean orderAudit;


    /**
     * 是否开启库存预警
     */
    private Boolean stockWarn;
    private Long stockWarnValue;


    /*-------------------- 购买限制-地域限制 --------------------*/


    /**
     * 是否有地域限制
     */
    private Boolean areaLimitSwitch;
    /**
     * 地域限制类型,1-白名单 2-黑名单
     *
     * @see AddrLimitTypeEnum
     */

    private Integer areaLimitType;

    /**
     * 地域限制
     * 格式 省code@市code
     */
    private List<String> areaLimitAreaCodes = new ArrayList<>();


    /*--------------------实物相关 物流信息 --------------------*/

    /**
     * 供应商备注
     */
    @Length(max = 60, message = "供应商备注过长")
    private String supplierRemark = "";
    /**
     * 运费类型
     * free(包邮),unity(统一运费),template(运费模板)
     */
    private String expressType = "unity";
    /**
     * 运费,单位分
     */
    @Min(value = 0, message = "运费不能小于0")
    private Long expressPrice = 0L;

    /**
     * 运费模版Id
     */
    private String expressTemplateId;
    /**
     * 待发货文案
     */
    @Length(max = 100, message = "待发货文案过长")
    private String expressAlertMessage;


    /**
     * 媒体信息,json存储，前端给啥存啥
     */
    @Setter
    @Getter
    public static class Media implements Serializable {
        private static final long serialVersionUID = 1L;
        /**
         * oss链接
         */
        private String url;
        /**
         * 媒体类型
         */
        private String type;
        /**
         * 文件名称
         */
        private String name;
        /**
         * 文件大小
         */
        private String size;

    }

    /**
     * 获取所有叶子上的skuInfo
     */
    public List<SkuAttributeNode.SkuInfo> leafSkuInfoList() {
        return SkuAttributeNode.leafSkuInfoList(new AttributeResolver.SkuInfoAttributeResolver().resolveNodes(attributes));
    }

    public void valid() throws BizException {
        for (SkuAttributeNode.SkuInfo skuInfo : leafSkuInfoList()) {
            skuInfo.valid();
            Long salePrice = skuInfo.getSalePrice();
            Conditions.expectTrue(skuInfo.getFacePrice() >= salePrice, "划线价不能小于供货价");
            if (skuInfo.getSuggestMarketPrice() != null) {
                Conditions.expectTrue(skuInfo.getSuggestMarketPrice() >= salePrice, "建议售价不能小于供货价");
            }
        }
    }


    /* ************************* 以下是充值相关 ************************* */

    /**
     * 是否需要填写账号
     */
    private Boolean fillAccount;
    /**
     * 标题
     */
    private String fillTitle;
    /**
     * 提示文案
     */
    private String fillTips;
    /**
     * 格式校验
     * 0-不限格式 1-手机号
     */
    private Integer accountMode;


    /* ************************* 以下是卡券、充值相关 ************************* */


    /**
     * 是否展示跳转按钮
     */
    private Boolean showJumpButton;
    /**
     * 跳转文案
     */
    private String jumpLink;
    /**
     * 跳转链接
     */
    private String jumpDesc;

    /* ************************* 以下是卡券相关 ************************* */


    /**
     * 发券方式 0-备货发券  1-人工后发券
     */
    @Range(message = "发券方式 0-备货发券  1-人工后发券", min = 0, max = 1)
    private Integer couponMode = 0;


    /**
     * 卡券使用规则
     */
    @Length(max = 3000, message = "使用规则过长")
    private String usageRule;


    /**
     * 是否重复卡券/链接
     */
    private Boolean repeatCoupon;
}