package cn.com.duiba.cloud.duiba.http.client.proxy;



import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.duiba.http.client.bo.HttpClientRequest;
import cn.com.duiba.cloud.duiba.http.client.config.HttpClientConfig;
import cn.com.duiba.cloud.duiba.http.client.context.HttpRequestContext;
import cn.com.duiba.cloud.duiba.http.client.enums.HttpRequestMethod;
import cn.com.duiba.cloud.duiba.http.client.exception.ParamException;
import cn.com.duiba.cloud.duiba.http.client.factorybean.HttpFactoryBean;
import cn.com.duiba.cloud.duiba.http.client.handler.analysis.method.AnalysisMethodParamHandlerManager;
import cn.com.duiba.cloud.duiba.http.client.handler.http.request.SetHttpParamHandlerManager;
import cn.com.duiba.cloud.duiba.http.client.handler.http.result.HttpClientResultHandlerManager;
import cn.com.duiba.cloud.duiba.http.client.interceptor.HttpClientInterceptor;
import cn.com.duiba.cloud.duiba.http.client.response.HttpClientResponse;
import cn.com.duiba.cloud.duiba.http.client.utils.SpringUtil;
import cn.com.duiba.cloud.duiba.http.client.utils.UrlUtil;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.List;
import java.util.Objects;

/**
 * 动态代理基类
 *
 * @author linzhou
 */
@Data
public abstract class AbstractHttpProxy implements HttpProxy, InvocationHandler {

    private HttpFactoryBean httpFactoryBean;
    private List<HttpClientInterceptor> httpClientInterceptorList;
    private HttpClientConfig httpClientConfig;

    @Override
    public <T> T newProxyInstance() {
        Class<?> clazz = httpFactoryBean.getType();
        return (T) Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class[]{clazz}, this);
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {

        HttpRequestContext context = new HttpRequestContext(httpFactoryBean, proxy, method, args);
        return sendHttp(context);
    }

    private Object sendHttp(HttpRequestContext context) throws Throwable {
        try {
            //解析参数
            analysisMethodParam(context);
            //设置httpUrl
            setHttpUrl(context);
            //执行httpBefore方法
            Object rlt = runHttpBefore(context);
            if (Objects.nonNull(rlt)) {
                return rlt;
            }
            HttpClientResponse response = doInvoke(context);
            response.setContext(context);
            rlt = HttpClientResultHandlerManager.getReturnObject(response);
            //执行httpAfter方法处理返回数据
            return runHttpAfter(response, rlt);
        } catch (Throwable throwable) {
            //执行异常拦截
            Object rlt = runHttpException(context, throwable);
            if (Objects.nonNull(rlt)) {
                return rlt;
            }
            throw throwable;
        }
    }


    protected abstract HttpClientResponse doInvoke(HttpRequestContext context) throws Throwable;


    /**
     * 执行httpAfter方法
     *
     * @param context
     * @param e
     */
    private Object runHttpException(HttpRequestContext context, Throwable e) throws Exception {
        Object rlt = null;
        for (HttpClientInterceptor httpClientInterceptor : getHttpClientInterceptorList()) {
            rlt = httpClientInterceptor.httpException(context, e);
        }
        return rlt;
    }

    /**
     * 是否是get方法
     *
     * @param context
     * @return
     */
    protected boolean isGet(HttpRequestContext context) {
        HttpRequestMethod httpRequestMethod = context.getHttpRequestMethod();
        return HttpRequestMethod.POST != httpRequestMethod;
    }


    /**
     * 解析方法参数
     *
     * @param context
     * @return
     */
    protected void analysisMethodParam(HttpRequestContext context) throws Exception {
        Object[] args = context.getArgs();
        Annotation[][] parameterAnnotations = context.getParameterAnnotations();
        HttpClientRequest result = new HttpClientRequest();

        if (args != null) {

            for (int i = 0; i < args.length; i++) {
                Object arg = args[i];
                if (arg == null) {
                    continue;
                }
                Annotation[] parameterAnnotation = parameterAnnotations[i];

                Object methodParam = AnalysisMethodParamHandlerManager.analysisMethodParam(arg, parameterAnnotation);
//                result.addMethodParam(methodParam);
                SetHttpParamHandlerManager.setHttpParam(result,methodParam);
            }
        }
        context.setParam(result);
    }

    /**
     * 获取请求地址
     *
     * @return
     */
    public void setHttpUrl(HttpRequestContext context) {
        if (StringUtils.isBlank(context.getHttpUrl())) {
            if (Objects.isNull(context.getParam()) || Objects.isNull(context.getHttpRequestMethod())) {
                throw new ParamException("数据异常,methodParamResult or httpRequestMethod is null");
            }
            String baseUrl = context.getBaseUrl(getConfig().getBaseUrl());
            if (isGet(context)
                    || context.isPostEntity()) {
                context.setHttpUrl(UrlUtil.getParamUrl(baseUrl, context.getNameValueParams()));
            } else {
                context.setHttpUrl(baseUrl);
            }
        }
    }

    protected HttpClientConfig getConfig() {
        if (Objects.isNull(httpClientConfig)) {
            httpClientConfig = SpringUtil.getBean(HttpClientConfig.class);
        }
        return httpClientConfig;
    }


    /**
     * 执行httpBefore方法
     *
     * @param context
     */
    private Object runHttpBefore(HttpRequestContext context) throws Exception {
        for (HttpClientInterceptor httpClientInterceptor : getHttpClientInterceptorList()) {
            Object rlt = httpClientInterceptor.httpBefore(context);
            if (Objects.nonNull(rlt)) {
                return rlt;
            }
        }
        return null;
    }

    /**
     * 执行httpAfter方法
     *
     * @param response
     */
    private Object runHttpAfter(HttpClientResponse response,Object rlt) throws Exception {
        for (HttpClientInterceptor httpClientInterceptor : getHttpClientInterceptorList()) {
            rlt = httpClientInterceptor.httpAfter(response,rlt);
        }
        return rlt;
    }

    protected List<HttpClientInterceptor> getHttpClientInterceptorList() {
        if (httpClientInterceptorList == null) {
            synchronized (AbstractHttpProxy.class) {
                if (httpClientInterceptorList == null) {
                    httpClientInterceptorList = SpringUtil.getBeanList(HttpClientInterceptor.class);
                }
            }
        }
        return httpClientInterceptorList;
    }
}
